package org.jbpm.gd.jpdl.ui.properties;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import org.eclipse.gef.commands.CommandStack;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.jbpm.gd.common.notation.AbstractNotationElement;
import org.jbpm.gd.jpdl.model.Assignment;
import org.jbpm.gd.jpdl.model.ProcessDefinition;
import org.jbpm.gd.jpdl.model.Swimlane;
import org.jbpm.gd.jpdl.ui.JpdlLabelProvider;
import org.jbpm.gd.jpdl.ui.command.AssignmentCreateCommand;
import org.jbpm.gd.jpdl.ui.command.AssignmentDeleteCommand;
import org.jbpm.gd.jpdl.ui.command.NamedElementSetNameCommand;
import org.jbpm.gd.jpdl.ui.graph.part.NotationElementGraphicalEditPart;
import org.jbpm.gd.jpdl.ui.outline.part.JpdlOutlineEditPart;

public class SwimlaneContainerSection
	extends AbstractPropertySection implements PropertyChangeListener, SelectionListener, FocusListener {
	
	private Composite detailsArea;	
	private Table swimlaneTable;	
	private Group swimlaneInfoGroup;
	private Label swimlaneNameLabel;
	private Text swimlaneNameText;
	private Label assignmentTypeLabel;
	private CCombo assignmentTypeCombo;
	private Group assignmentInfoGroup;
	private AssignmentInformationPageBook assignmentInfoPageBook;	
    private ProcessDefinition processDefinition;
    private Swimlane selectedSwimlane;
    private String assignmentTypeComboSelection = "";    
    private TabbedPropertySheetPage tabbedPropertySheetPage;    
    private SwimlaneContainerSectionActionBarContributor actionBarContributor;
    
	public void createControls(Composite parent,
			TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);	
		actionBarContributor = new SwimlaneContainerSectionActionBarContributor(this);
		tabbedPropertySheetPage = aTabbedPropertySheetPage;
		final Composite composite = getWidgetFactory().createFlatFormComposite(parent);		
		createMasterArea(composite);
		createDetailsArea(composite);
	}
	
	public TabbedPropertySheetPage getTabbedPropertySheetPage() {
		return tabbedPropertySheetPage;
	}

	private void createMasterArea(Composite composite) {
		swimlaneTable = getWidgetFactory().createTable(
				composite, SWT.SINGLE | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
		swimlaneTable.setLayoutData(createSwimlaneListLayoutData());
		swimlaneTable.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleSwimlaneListSelection();
			}			
		});
		actionBarContributor.createPopupMenu(swimlaneTable);
	}
	
	private void handleSwimlaneListSelection() {
		if (processDefinition == null) return;
		int i = swimlaneTable.getSelectionIndex();
		if (i == -1) {
			setSelectedSwimlane(null);
		} else {
			setSelectedSwimlane((Swimlane)swimlaneTable.getSelection()[0].getData());
		}
	}

	private FormData createSwimlaneListLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 5);
		data.right = new FormAttachment(20, -5);
		data.top = new FormAttachment(0, 5);
		data.bottom = new FormAttachment(100, -5);
		return data;
	}
	
	private void createDetailsArea(Composite composite) {
		detailsArea = getWidgetFactory().createComposite(composite);
		detailsArea.setLayout(new FormLayout());
		detailsArea.setLayoutData(createDetailsAreaLayoutData());		
		createSwimlaneInfoArea();
		createAssignmentInfoArea();
	}

	private void createAssignmentInfoArea() {
		assignmentInfoGroup = getWidgetFactory().createGroup(detailsArea, "Assignment Information");
		assignmentInfoGroup.setLayout(new FormLayout());
		assignmentInfoPageBook = AssignmentInformationPageBook.create(getWidgetFactory(), assignmentInfoGroup);
		assignmentInfoGroup.setLayoutData(createAssignmentInfoGroupLayoutData());
	}
	
	private void createSwimlaneInfoArea() {
		swimlaneInfoGroup = getWidgetFactory().createGroup(detailsArea, "Swimlane Information");	
		swimlaneInfoGroup.setLayout(new FormLayout());
		swimlaneNameLabel = getWidgetFactory().createLabel(swimlaneInfoGroup, "Swimlane Name");
		swimlaneNameText = getWidgetFactory().createText(swimlaneInfoGroup, "");
		assignmentTypeLabel = getWidgetFactory().createLabel(swimlaneInfoGroup, "Assignment Type");
		assignmentTypeCombo = getWidgetFactory().createCCombo(swimlaneInfoGroup, SWT.READ_ONLY | SWT.BORDER);
		assignmentTypeCombo.setItems(getAssignmentTypes());
		assignmentTypeCombo.select(0);
		swimlaneInfoGroup.setLayoutData(createSwimlaneInfoGroupLayoutData());
		swimlaneNameLabel.setLayoutData(createSwimlaneNameLabelLayoutData());
		swimlaneNameText.setLayoutData(createSwimlaneNameTextLayoutData());
		assignmentTypeLabel.setLayoutData(createAssignmentTypeLabelLayoutData());
		assignmentTypeCombo.setLayoutData(createAssignmentTypeComboLayoutData());
	}
	
	private void updateSwimlaneName() {
		NamedElementSetNameCommand command = new NamedElementSetNameCommand();
		command.setName(swimlaneNameText.getText());
		command.setNamedElement(selectedSwimlane);
		getCommandStack().execute(command);
	}
	
	private void updateAssignmentType() {
		String text = assignmentTypeCombo.getText();
		if (assignmentTypeComboSelection.equals(text)) return;
		assignmentTypeComboSelection = text;
		if ("".equals(text)) {
			getCommandStack().execute(new AssignmentDeleteCommand(selectedSwimlane));
		} else {
			if (selectedSwimlane.getAssignment() == null) {
				getCommandStack().execute(new AssignmentCreateCommand(selectedSwimlane)); 
		    }
		}
		assignmentInfoPageBook.updateAssignmentInfo(text);
	}	
	
	private FormData createDetailsAreaLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(swimlaneTable, 5);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(0, 0);
		data.bottom = new FormAttachment(100, 0);
		return data;
	}
	
	private FormData createSwimlaneNameLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 5);
		data.bottom = new FormAttachment(swimlaneNameText, -4);
		data.bottom.alignment = SWT.BOTTOM;
		return data;
	}
	
	private FormData createSwimlaneNameTextLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(swimlaneNameLabel, 5);
		data.top = new FormAttachment(0, 5);
		data.right = new FormAttachment(50, -5);
		data.bottom = new FormAttachment(100, -5);
		return data;
	}
	
	private FormData createAssignmentTypeLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(50, 5);
		data.bottom = new FormAttachment(swimlaneNameText, -4);
		data.bottom.alignment = SWT.BOTTOM;
		return data;
	}
	
	private FormData createAssignmentTypeComboLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(assignmentTypeLabel, 5);
		data.top = new FormAttachment(0, 5);
		data.right = new FormAttachment(100, -5);
		data.bottom = new FormAttachment(100, -5);
		return data;
	}
	
	private FormData createSwimlaneInfoGroupLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 4);
		data.top = new FormAttachment(0, 4);
		data.right = new FormAttachment(100, -4);
		return data;
	}
	
	private FormData createAssignmentInfoGroupLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 4);
		data.top = new FormAttachment(swimlaneInfoGroup, 4);
		data.right = new FormAttachment(100, -4);
		data.bottom = new FormAttachment(100, -4);
		return data;
	}
	
	private String[] getAssignmentTypes() {
		return new String[] { "", "Expression", "Actor", "Pooled Actors", "Handler" };
	}

	public boolean shouldUseExtraSpace() {
		return true;
	}
	
	public void aboutToBeShown() {
		actionBarContributor.activateContributions();
	}
	
	public void aboutToBeHidden() {
		actionBarContributor.deactivateContributions();
	}
	
 	public void setInput(IWorkbenchPart part, ISelection selection) {
        super.setInput(part, selection);
        if (!(selection instanceof IStructuredSelection)) return;
        Object input = ((IStructuredSelection)selection).getFirstElement();
        if (input instanceof NotationElementGraphicalEditPart) {
        	AbstractNotationElement notationElement = ((NotationElementGraphicalEditPart)input).getNotationElement();
        	setProcessDefinition((ProcessDefinition)notationElement.getSemanticElement());
        } else if (input instanceof JpdlOutlineEditPart) {
        	setProcessDefinition((ProcessDefinition)((JpdlOutlineEditPart)input).getModel());
        }
    }
 	
 	public void clearControls() {
 		setSelectedSwimlane(null);
 		swimlaneTable.removeAll();
 		clearSwimlaneInfo();
 		clearAssignmentInfo();
 	}
 	
 	private void clearSwimlaneInfo() {
 		swimlaneNameText.setText("");
 		swimlaneNameText.setEnabled(false);
 		assignmentTypeCombo.setText(""); 
 		assignmentTypeCombo.setEnabled(false);
 	}
 	
 	private void clearAssignmentInfo() {
 		assignmentInfoPageBook.clear();
 	}
 	
 	public void setProcessDefinition(ProcessDefinition newProcessDefinition) {
 		if (processDefinition == newProcessDefinition) return;
 		if (processDefinition != null) {
 			processDefinition.removePropertyChangeListener(this);
 		}
 		clearControls();
 		processDefinition = newProcessDefinition;
 		if (processDefinition != null) {
 			updateControls();
	 		processDefinition.addPropertyChangeListener(this);
 		}
 	}
 	
 	private void updateControls() {
 		Swimlane[] swimlanes = processDefinition.getSwimlanes();
 		for (int i = 0; i < swimlanes.length; i++) {
 			TableItem tableItem = new TableItem(swimlaneTable, SWT.NULL);
 			tableItem.setText(swimlanes[i].getName());
 			tableItem.setData(swimlanes[i]);
 			tableItem.setImage(JpdlLabelProvider.getImage(swimlanes[i]));
 		}
 	}
 	
 	public ProcessDefinition getProcessDefinition() {
 		return processDefinition;
 	}
 	
 	private void hookListeners() {
 		swimlaneNameText.addSelectionListener(this);
 		swimlaneNameText.addFocusListener(this);
 		assignmentTypeCombo.addSelectionListener(this);
 	}
 	
 	private void unhookListeners() {
 		swimlaneNameText.removeSelectionListener(this);
 		swimlaneNameText.removeFocusListener(this);
 		assignmentTypeCombo.removeSelectionListener(this); 		
 	}
 	
 	private void setSelectedSwimlane(Swimlane swimlane) {
 		unhookListeners();
        if (selectedSwimlane != null) {
        	selectedSwimlane.removePropertyChangeListener(this);
        }
 		selectedSwimlane = swimlane;
 		updateSwimlaneInfoGroup();
 		updateAssignmentInfoGroup();
 		if (selectedSwimlane != null) {
 			selectedSwimlane.addPropertyChangeListener(this);
 	 		hookListeners();
 		}
 		actionBarContributor.setRemoveSwimlaneEnabled(selectedSwimlane != null);
 	}
 	
 	private void updateAssignmentInfoGroup() {
 		Assignment assignment = null;
 		if (selectedSwimlane != null) {
 			assignment = selectedSwimlane.getAssignment();
 		}
 		assignmentInfoPageBook.setAssignment(assignment);
 	}
 	
 	private void updateSwimlaneInfoGroup() { 		
 		updateSwimlaneNameText();
		updateAssignmentTypeCombo();
 	}

	private void updateSwimlaneNameText() {
		swimlaneNameText.setEnabled(selectedSwimlane != null);
		swimlaneNameText.setText(selectedSwimlane == null ? "" : selectedSwimlane.getName());
	}
 	
 	private void updateAssignmentTypeCombo() {
 		if (selectedSwimlane == null) {
 			assignmentTypeCombo.setText("");
 			assignmentTypeCombo.setEnabled(false);
 		} else {
	 		Assignment assignment = selectedSwimlane.getAssignment();
	 		if (assignment == null) {
	 			assignmentTypeCombo.setText("");
	 		} else if (assignment.getExpression() != null) {
	 			assignmentTypeCombo.setText("Expression");
	 		} else if (assignment.getActorId() != null) {
	 			assignmentTypeCombo.setText("Actor");
	 		} else if (assignment.getPooledActors() != null) {
	 			assignmentTypeCombo.setText("Pooled Actors");
	 		} else if (assignment.getClassName() != null) {
	 			assignmentTypeCombo.setText("Handler");
	 		}
	 		assignmentTypeCombo.setEnabled(true);
 		}
 	}
 	
 	public Swimlane getSelectedSwimlane() {
 		return selectedSwimlane;
 	}

	public void propertyChange(PropertyChangeEvent evt) {
		if (swimlaneTable.isDisposed()) return;
		if ("swimlaneAdd".equals(evt.getPropertyName())) {
			TableItem tableItem = new TableItem(swimlaneTable, SWT.NULL);
			Swimlane swimlane = (Swimlane)evt.getNewValue();
			tableItem.setText(swimlane.getName());
			tableItem.setData(swimlane);
			tableItem.setImage(JpdlLabelProvider.getImage(swimlane));
			swimlaneTable.setSelection(tableItem);
			swimlaneTable.notifyListeners(SWT.Selection, new Event());
		} else if ("swimlaneRemove".equals(evt.getPropertyName())) {
			TableItem tableItem = getItemToRemove(evt.getOldValue());
			if (tableItem != null) {
				tableItem.dispose();
				swimlaneTable.notifyListeners(SWT.Selection, new Event());
			}
		} else if ("name".equals(evt.getPropertyName())) {
			TableItem tableItem = swimlaneTable.getItem(swimlaneTable.getSelectionIndex());
			tableItem.setText((String)evt.getNewValue());
		} else if ("assignment".equals(evt.getPropertyName())) {
			if (selectedSwimlane != null) {
				assignmentInfoPageBook.setAssignment(selectedSwimlane.getAssignment());
			}
		}
	}
	
	private TableItem getItemToRemove(Object object) {
		for (int i = 0; i < swimlaneTable.getItemCount(); i++) {
			if (swimlaneTable.getItem(i).getData() == object)
				return swimlaneTable.getItem(i);
		}
		return null;
	}
	
	private CommandStack getCommandStack() {
		return (CommandStack)PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getActiveEditor().getAdapter(CommandStack.class);
	}

	public void widgetDefaultSelected(SelectionEvent e) {
		if (e.widget == swimlaneNameText) {
			updateSwimlaneName();
		}
	}

	public void widgetSelected(SelectionEvent e) {
		if (e.widget == assignmentTypeCombo) {
			updateAssignmentType();
		}
	}

	public void focusGained(FocusEvent e) {
	}

	public void focusLost(FocusEvent e) {
		if (e.widget == swimlaneNameText) {
			updateSwimlaneName();
		}
	}
	
}
