/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.tools.ant;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipInputStream;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import org.jbpm.bpel.graph.def.BpelProcessDefinition;
import org.jbpm.bpel.tools.WsdlServiceTool;
import org.jbpm.bpel.xml.ProblemCollector;
import org.jbpm.jpdl.par.ProcessArchive;
import org.jbpm.jpdl.xml.Problem;

/**
 * Task adapter for the {@link WsdlServiceTool wsdlservice} tool.
 * @author Alejandro Guizar
 * @version $Revision: 1.4 $ $Date: 2007/09/10 23:44:45 $
 */
public class GenerateWsdlServiceTask extends Task {

  private File processfile;
  private File outputdir;
  private String servicefilename;
  private String bindingfilesprefix;
  private String bindingfilessuffix;

  
  public void execute() throws BuildException {
    // read process archive
    ProcessArchive archive;
    try {
      archive = readProcessArchive();
    }
    catch (IOException e) {
      throw new BuildException("could not read process archive: " + processfile, e);
    }

    // parse process definition
    BpelProcessDefinition definition = (BpelProcessDefinition) archive.parseProcessDefinition();

    // halt on parser errors
    if (Problem.containsProblemsOfLevel(archive.getProblems(), Problem.LEVEL_ERROR))
      throw new BuildException("process definition is invalid");

    WsdlServiceTool tool = new WsdlServiceTool();

    // output directory - optional
    if (outputdir != null)
      tool.setWsdlDirectory(outputdir);

    // binding file prefix - optional
    if (bindingfilesprefix != null)
      tool.setBindingFilesPrefix(bindingfilesprefix);

    // binding file suffix - optional
    if (bindingfilessuffix != null)
      tool.setBindingFilesSuffix(bindingfilessuffix);

    // service file name - optional
    if (servicefilename != null)
      tool.setServiceFileName(servicefilename);

    ProblemCollector collector = new ProblemCollector();
    tool.setProblemHandler(collector);

    try {
      tool.generateWsdlService(definition);
    }
    catch (RuntimeException e) {
      /*
       * ant does not show stack traces of exceptions thrown from tasks, making it extremely
       * difficult to debug unexpected exceptions;
       */
      e.printStackTrace();
      throw new BuildException(e.toString());
    }

    // halt on generator errors
    if (Problem.containsProblemsOfLevel(collector.getProblems(), Problem.LEVEL_ERROR))
      throw new BuildException("could not generate service definition");
  }

  private ProcessArchive readProcessArchive() throws IOException {
    InputStream processStream = new FileInputStream(processfile);
    try {
      ProcessArchive processArchive = new ProcessArchive(new ZipInputStream(processStream));
      return processArchive;
    }
    finally {
      processStream.close();
    }
  }

  public void setProcessfile(File processfile) {
    this.processfile = processfile;
  }

  public void setOutputdir(File outputdir) {
    this.outputdir = outputdir;
  }

  public void setServicefilename(String serviceFile) {
    this.servicefilename = serviceFile;
  }

  public void setBindingfilesprefix(String bindingFile) {
    this.bindingfilesprefix = bindingFile;
  }

  public void setBindingfilessuffix(String bindingfilessuffix) {
    this.bindingfilessuffix = bindingfilessuffix;
  }
}
