/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.persistence.db;

import org.hibernate.Criteria;
import org.hibernate.LockMode;
import org.hibernate.Session;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import org.jbpm.JbpmContext;
import org.jbpm.bpel.BpelException;
import org.jbpm.bpel.graph.def.BpelProcessDefinition;
import org.jbpm.graph.exe.Token;
import org.jbpm.persistence.PersistenceService;

/**
 * @author Alejandro Guizar
 * @version $Revision: 1.3 $ $Date: 2007/07/20 21:46:36 $
 */
public class BpelGraphSession {

  private Session session;

  public BpelGraphSession(Session session) {
    this.session = session;
  }

  public BpelProcessDefinition findProcessDefinition(String name, String targetNamespace,
      int version) {
    Criteria criteria = session.createCriteria(BpelProcessDefinition.class)
      .add(Restrictions.eq("name", name))
      .add(Restrictions.eq("version", new Integer(version)));

    if (targetNamespace != null)
      criteria.add(Restrictions.eq("targetNamespace", targetNamespace));

    return (BpelProcessDefinition) criteria.uniqueResult();
  }

  public BpelProcessDefinition findLatestProcessDefinition(String name, String targetNamespace) {
    Criteria criteria = session.createCriteria(BpelProcessDefinition.class)
      .add(Restrictions.eq("name", name))
      .addOrder(Order.desc("version"))
      .setMaxResults(1);

    if (targetNamespace != null)
      criteria.add(Restrictions.eq("targetNamespace", targetNamespace));

    return (BpelProcessDefinition) criteria.uniqueResult();
  }

  public void deployProcessDefinition(BpelProcessDefinition processDefinition) {
    String processName = processDefinition.getName();
    if (processName == null)
      throw new BpelException("process definition has no name");

    BpelProcessDefinition latestProcessDefinition = findLatestProcessDefinition(processName,
        processDefinition.getTargetNamespace());
    // find the current latest process definition
    if (latestProcessDefinition != null) {
      // take the next version number
      processDefinition.setVersion(latestProcessDefinition.getVersion() + 1);
    }
    else {
      // start from 1
      processDefinition.setVersion(1);
    }

    session.save(processDefinition);
  }

  public void lockToken(Token token) {
    session.lock(token, LockMode.UPGRADE);
  }

  public static BpelGraphSession getInstance(JbpmContext jbpmContext) {
    PersistenceService persistenceService = jbpmContext.getServices().getPersistenceService();
    if (persistenceService instanceof DbPersistenceService) {
      DbPersistenceService dbPersistenceService = (DbPersistenceService) persistenceService;
      return dbPersistenceService.getBpelGraphSession();
    }
    return null;
  }
}
