/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.integration.jms;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.jms.JMSException;
import javax.jms.Session;
import javax.wsdl.Operation;
import javax.wsdl.Port;
import javax.wsdl.extensions.soap.SOAPAddress;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ibm.wsdl.extensions.soap.SOAPConstants;

import org.jbpm.JbpmContext;
import org.jbpm.bpel.BpelException;
import org.jbpm.bpel.endpointref.EndpointReference;
import org.jbpm.bpel.graph.def.BpelProcessDefinition;
import org.jbpm.bpel.graph.exe.BpelFaultException;
import org.jbpm.bpel.integration.IntegrationService;
import org.jbpm.bpel.integration.catalog.DefinitionCatalog;
import org.jbpm.bpel.integration.client.SoapClient;
import org.jbpm.bpel.integration.def.InvokeAction;
import org.jbpm.bpel.integration.def.PartnerLinkDefinition;
import org.jbpm.bpel.integration.def.ReceiveAction;
import org.jbpm.bpel.integration.def.ReplyAction;
import org.jbpm.bpel.integration.exe.PartnerLinkInstance;
import org.jbpm.bpel.variable.def.MessageType;
import org.jbpm.bpel.variable.exe.MessageValue;
import org.jbpm.bpel.wsdl.xml.WsdlUtil;
import org.jbpm.graph.exe.Token;

/**
 * @author Alejandro Guzar
 * @version $Revision: 1.18 $ $Date: 2007/09/04 06:36:16 $
 */
public class JmsIntegrationService implements IntegrationService {

  private final JmsIntegrationServiceFactory factory;

  private List requestListeners = new ArrayList();

  private static final Log log = LogFactory.getLog(JmsIntegrationService.class);
  private static final long serialVersionUID = 1L;

  JmsIntegrationService(JmsIntegrationServiceFactory factory) {
    this.factory = factory;
  }

  public void receive(ReceiveAction receiveAction, Token token, boolean oneShot) {
    IntegrationControl integrationControl = getIntegrationControl(token);
    try {
      jmsReceive(receiveAction, token, integrationControl, oneShot);
    }
    catch (JMSException e) {
      throw new BpelException("could not create request listener", e);
    }
  }

  void jmsReceive(ReceiveAction receiveAction, Token token, IntegrationControl integrationControl,
      boolean oneShot) throws JMSException {
    Session jmsSession = createJmsSession(integrationControl);
    RequestListener requestListener = new RequestListener(receiveAction, token, integrationControl,
        jmsSession, oneShot);
    requestListeners.add(requestListener);
  }

  public void receive(List receivers, Token token) {
    IntegrationControl integrationControl = getIntegrationControl(token);
    try {
      jmsReceive(receivers, token, integrationControl);
    }
    catch (JMSException e) {
      throw new BpelException("could not create request listeners", e);
    }
  }

  void jmsReceive(List receivers, Token token, IntegrationControl integrationControl)
      throws JMSException {
    Session jmsSession = createJmsSession(integrationControl);
    Iterator receiverIt = receivers.iterator();
    while (receiverIt.hasNext()) {
      ReceiveAction receiveAction = (ReceiveAction) receiverIt.next();
      RequestListener requestListener = new RequestListener(receiveAction, token,
          integrationControl, jmsSession);
      requestListeners.add(requestListener);
    }
  }

  private static Session createJmsSession(IntegrationControl integrationControl)
      throws JMSException {
    return integrationControl.getJmsConnection().createSession(false, Session.CLIENT_ACKNOWLEDGE);
  }

  public void cancelReception(ReceiveAction receiveAction, Token token) {
    try {
      jmsCancelReception(receiveAction, token, getIntegrationControl(token));
    }
    catch (JMSException e) {
      log.debug("could not close request listener", e);
    }
  }

  void jmsCancelReception(ReceiveAction receiveAction, Token token,
      IntegrationControl integrationControl) throws JMSException {
    RequestListener requestListener = integrationControl.removeRequestListener(receiveAction, token);
    // some competing thread might have removed the request listener already
    if (requestListener == null)
      return;
    /*
     * this service may not have created the request listener, but removing it in case it did is
     * necessary to avoid opening it later on when this service closes
     */
    requestListeners.remove(requestListener);
    // release listener resources
    requestListener.close();
  }

  public void reply(ReplyAction replyAction, Token token) {
    try {
      replyOutstandingRequest(replyAction, token);
    }
    catch (JMSException e) {
      throw new BpelException("could not send reply", e);
    }
  }

  private void replyOutstandingRequest(ReplyAction replyAction, Token token) throws JMSException {
    // extract the output parts
    Map parts = replyAction.writeMessage(token);

    // obtain the outstanding request for the partner link, operation and
    // message exchange of the replier
    IntegrationControl integrationControl = getIntegrationControl(token);
    OutstandingRequest request = integrationControl.removeOutstandingRequest(replyAction, token);

    Session jmsSession = createJmsSession(integrationControl);
    try {
      request.sendReply(parts, replyAction.getFaultName(), jmsSession);
    }
    finally {
      jmsSession.close();
    }
  }

  public void invoke(InvokeAction invokeAction, Token token) {
    // extract the input parts
    Map inputParts = invokeAction.writeMessage(token);

    PartnerLinkDefinition partnerLinkDef = invokeAction.getPartnerLink();
    PartnerLinkInstance partnerLinkInst = partnerLinkDef.getInstance(token);

    // acquire a client for the partner link
    IntegrationControl integrationControl = getIntegrationControl(token);
    SoapClient partnerClient = integrationControl.getPartnerClient(partnerLinkInst);

    Operation operation = invokeAction.getOperation();
    String operationName = operation.getName();

    // is this a request/response operation?
    if (operation.getOutput() != null) {
      // send input, block for output
      Map outputParts;
      try {
        outputParts = partnerClient.call(operationName, inputParts);
      }
      catch (BpelFaultException e) {
        replaceMessageType(e, token);
        throw e;
      }

      log.debug("invoked: partnerLink="
          + partnerLinkDef.getName()
          + ", operation="
          + operationName
          + ", output="
          + outputParts);

      // assign the output data
      invokeAction.readMessage(token, outputParts);
    }
    else {
      // fire and forget
      partnerClient.callOneWay(operationName, inputParts);
      log.debug("invoked: partnerLink=" + partnerLinkDef.getName() + ", operation=" + operationName);
    }
  }

  /**
   * Replaces the transient message type in the given fault exception with the persistent object
   * from the process definition. {@link SoapClient} produces faults with transient message types.
   */
  private static void replaceMessageType(BpelFaultException faultException, Token token) {
    // extract the message value from the exception
    MessageValue faultData = faultException.getFaultInstance().getMessageValue();

    // find the related message type
    BpelProcessDefinition processDefinition = (BpelProcessDefinition) token.getProcessInstance()
        .getProcessDefinition();
    MessageType persistentType = processDefinition.getImportDefinition().getMessageType(
        faultData.getType().getName());

    // replace with the persistent object
    faultData.setType(persistentType);
  }

  public EndpointReference getMyReference(PartnerLinkDefinition partnerLink, Token token) {
    IntegrationControl integrationControl = getIntegrationControl(token);
    EndpointReference myReference = integrationControl.getPartnerLinkEntry(partnerLink)
        .getMyReference();
    // fill in address, if missing
    if (myReference.getAddress() == null) {
      DefinitionCatalog catalog = new DefinitionCatalog();
      catalog.addDefinition(integrationControl.getWsdlDefinition());

      Port myPort = myReference.selectPort(catalog);
      SOAPAddress soapAddress = (SOAPAddress) WsdlUtil.getExtension(
          myPort.getExtensibilityElements(), SOAPConstants.Q_ELEM_SOAP_ADDRESS);
      if (soapAddress != null)
        myReference.setAddress(soapAddress.getLocationURI());
    }
    return myReference;
  }

  public void close() {
    JbpmContext jbpmContext = factory.getJbpmConfiguration().getCurrentJbpmContext();
    // open request listeners only if transaction is not marked for rollback
    if (!jbpmContext.getServices().getTxService().isRollbackOnly()) {
      try {
        openRequestListeners();
      }
      catch (JMSException e) {
        throw new BpelException("could not open subsequent request listeners", e);
      }
    }
    else
      closeRequestListeners();
  }

  private void openRequestListeners() throws JMSException {
    for (int i = 0, n = requestListeners.size(); i < n; i++) {
      RequestListener requestListener = (RequestListener) requestListeners.get(i);
      requestListener.open();
    }
  }

  private void closeRequestListeners() {
    for (int i = 0, n = requestListeners.size(); i < n; i++) {
      RequestListener requestListener = (RequestListener) requestListeners.get(i);
      try {
        requestListener.close();
      }
      catch (JMSException e) {
        log.debug("could not close request listener", e);
      }
    }
  }

  IntegrationControl getIntegrationControl(Token token) {
    return factory.getIntegrationControl(token.getProcessInstance().getProcessDefinition());
  }

  public static JmsIntegrationService get(JbpmContext jbpmContext) {
    return (JmsIntegrationService) jbpmContext.getServices().getService(
        IntegrationService.SERVICE_NAME);
  }
}
