/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.graph.struct;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.jbpm.JbpmContext;
import org.jbpm.bpel.graph.def.Activity;
import org.jbpm.bpel.graph.def.BpelVisitor;
import org.jbpm.bpel.graph.def.LinkDefinition;
import org.jbpm.bpel.persistence.db.BpelGraphSession;
import org.jbpm.graph.exe.ExecutionContext;
import org.jbpm.graph.exe.Token;

/**
 * Specifies one or more activities to be performed concurrently. {@link LinkDefinition links} can
 * be used within a <tt>flow</tt> to define explicit control dependencies between nested child
 * activities.
 * @author Juan Cant
 * @version $Revision: 1.13 $ $Date: 2007/07/26 00:39:10 $
 */
public class Flow extends StructuredActivity {

  private Map links = new HashMap();

  private static final Log log = LogFactory.getLog(Flow.class);
  private static final long serialVersionUID = 1L;

  public Flow() {
  }

  public Flow(String name) {
    super(name);
  }

  // behaviour methods
  // /////////////////////////////////////////////////////////////////////////////

  public void execute(ExecutionContext exeContext) {
    // initialize links
    Token flowToken = initializeLinks(exeContext.getToken());

    // create concurrent tokens
    Token[] concurrentTokens = createConcurrentTokens(flowToken);

    // execute concurrent tokens on activities
    List activities = getNodes();
    for (int i = 0, n = concurrentTokens.length; i < n; i++) {
      Activity activity = (Activity) activities.get(i);
      Token concurrentToken = concurrentTokens[i];

      begin.leave(new ExecutionContext(concurrentToken), activity.getDefaultArrivingTransition());

      // stop if flow token is prematurely ended
      if (flowToken.hasEnded())
        break;
    }
  }

  public Token initializeLinks(Token token) {
    if (!links.isEmpty()) {
      // a new token is required to contain the link variables
      token = new Token(token, getName());

      Iterator linkIt = links.values().iterator();
      while (linkIt.hasNext()) {
        LinkDefinition link = (LinkDefinition) linkIt.next();
        link.createInstance(token);
      }
    }
    return token;
  }

  public Token[] createConcurrentTokens(Token parent) {
    List activities = getNodes();
    int activityCount = activities.size();
    Token[] tokens = new Token[activityCount];

    for (int i = 0; i < activityCount; i++) {
      Activity activity = (Activity) activities.get(i);
      tokens[i] = new Token(parent, activity.getName());
    }
    return tokens;
  }

  public void terminate(ExecutionContext exeContext) {
    Iterator concurrentTokenIt = exeContext.getToken().getChildren().values().iterator();
    // terminate the active children
    while (concurrentTokenIt.hasNext()) {
      Token concurrentToken = (Token) concurrentTokenIt.next();
      if (concurrentToken.isAbleToReactivateParent()) {
        // cancel atomic activities
        Activity activity = (Activity) concurrentToken.getNode();
        activity.terminate(new ExecutionContext(concurrentToken));
      }
      // end concurrent token (do not verify parent termination)
      concurrentToken.end(false);
    }
  }

  public void leave(ExecutionContext exeContext) {
    Token token = exeContext.getToken();

    // mark child token as having reached the meeting point
    token.setAbleToReactivateParent(false);

    if (mustReactivateParent(token)) {
      Token parentToken = token.getParent();
      // if a token was created for scoping links, end it
      if (!links.isEmpty()) {
        parentToken.end(false);
        parentToken = parentToken.getParent();
      }
      getEnd().leave(new ExecutionContext(parentToken));
    }
  }

  public void eliminatePath(Token token) {
    /*
     * create link instances before eliminating path! override this method instead of skip() to fix
     * branch elimination (e.g. <if> and <pick>) as well
     */
    token = initializeLinks(token);
    super.eliminatePath(token);
  }

  protected boolean mustReactivateParent(Token token) {
    BpelGraphSession bpelGraphSession = BpelGraphSession.getInstance(JbpmContext.getCurrentJbpmContext());

    for (Iterator i = token.getParent().getChildren().values().iterator(); i.hasNext();) {
      Token concurrentToken = (Token) i.next();
      if (concurrentToken == token)
        continue;
      /*
       * acquire lock on the concurrent token to prevent race conditions between siblings where all
       * of them determine the parent must not be reactivated yet
       */
      if (bpelGraphSession != null)
        bpelGraphSession.lockToken(concurrentToken);

      if (concurrentToken.isAbleToReactivateParent()) {
        log.debug("flow will not reactivate parent yet, found concurrent token: " + concurrentToken);
        return false;
      }
    }
    return true;
  }

  protected boolean isChildInitial(Activity child) {
    /*
     * this method is only invoked from child.isInitial() on its composite activity; therefore, it
     * is valid to assume the argument is one of the concurrent activities of this flow
     */
    return true;
  }

  // LinkDefinition methods
  // /////////////////////////////////////////////////////////////////////////////

  public LinkDefinition findLink(String name) {
    LinkDefinition link = getLink(name);
    return link != null ? link : super.findLink(name);
  }

  public void addLink(LinkDefinition link) {
    links.put(link.getName(), link);
  }

  public Map getLinks() {
    return links;
  }

  public LinkDefinition getLink(String linkName) {
    return (LinkDefinition) links.get(linkName);
  }

  public void accept(BpelVisitor visitor) {
    visitor.visit(this);
  }
}
