/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.graph.exe.state;

import org.jbpm.bpel.graph.exe.ScopeInstance;
import org.jbpm.bpel.graph.exe.ScopeState;

/**
 * @author Juan Cant
 * @version $Revision: 1.6 $ $Date: 2007/03/22 13:32:55 $
 */
public class EndedState extends ScopeState {

  private static final long serialVersionUID = 1L;

  public static final ScopeState TERMINATED = new EndedState("terminated", 13);

  public static final ScopeState COMPENSATED = new EndedState("compensated", 14);

  public static final ScopeState COMPLETED = new EndedState("completed", 15) {

    private static final long serialVersionUID = 1L;

    public void compensate(ScopeInstance scope) {
      CompensatingState.enterCompensating(scope);
    }
  };

  public static final ScopeState FAULTED = new EndedState("faulted", 17) {

    private static final long serialVersionUID = 1L;

    public void terminate(ScopeInstance scope) {
      /*
       * in case a Terminate signal is sent to a nested scope that has already
       * faulted internally, the Terminate signal is ignored; the scope will
       * eventually send either a Faulted or an Exited signal to the parent
       */
    }
  };

  public static final ScopeState EXITED = new EndedState("exited", 18);

  private EndedState(String name, int code) {
    super(name, code);
  }

  public static void enterCompleted(ScopeInstance scope) {
    scope.setState(COMPLETED);
    scope.close();
  }

  public static void enterFaulted(ScopeInstance scope) {
    scope.setState(FAULTED);
    // end scope token (do not verify parent completion)
    scope.getToken().end(false);

    // rethrow fault to parent scope
    ScopeInstance parentInstance = scope.getParent();
    if (parentInstance != null)
      parentInstance.faulted(scope.getFaultInstance());
  }

  public static void enterCompensated(ScopeInstance scope) {
    scope.setState(COMPENSATED);
    scope.getCompensator().scopeCompensated(scope);
  }

  public static void enterTerminated(ScopeInstance scope) {
    scope.setState(TERMINATED);

    ScopeInstance parent = scope.getParent();
    if (parent != null)
      parent.scopeTerminated(scope);
  }

  public static void enterExited(ScopeInstance scope) {
    scope.setState(EndedState.EXITED);
    scope.close();
  }
}
