/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.graph.def;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;

import org.hibernate.Session;

import org.jbpm.JbpmContext;
import org.jbpm.bpel.graph.exe.BpelFaultException;
import org.jbpm.bpel.graph.scope.Scope;
import org.jbpm.bpel.sublang.def.JoinCondition;
import org.jbpm.bpel.xml.BpelConstants;
import org.jbpm.bpel.xml.util.DatatypeUtil;
import org.jbpm.graph.def.Event;
import org.jbpm.graph.def.GraphElement;
import org.jbpm.graph.def.Node;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.def.Transition;
import org.jbpm.graph.exe.ExecutionContext;
import org.jbpm.graph.exe.Token;

/**
 * Activities perform the process logic. They are divided into 2 classes: basic and structured.
 * @author Juan Cant
 * @version $Revision: 1.15 $ $Date: 2007/07/26 11:19:17 $
 */
public abstract class Activity extends Node {

  private CompositeActivity compositeActivity;

  private List sources = new ArrayList();
  private List targets = new ArrayList();

  private JoinCondition joinCondition;
  private Boolean suppressJoinFailure;

  private boolean unnamed;

  protected Activity() {
  }

  protected Activity(String name) {
    // at construction time, setName() cannot do any meaningful check
    this.name = name;
  }

  public void accept(BpelVisitor visitor) {
    // ignore call in non-visitable activities
  }

  // jbpm override
  // /////////////////////////////////////////////////////////////////////////////

  public void setName(String name) {
    if (compositeActivity != null && compositeActivity.hasNode(name))
      throw new IllegalArgumentException("duplicate name: " + name);

    this.name = name;
  }

  public boolean isUnnamed() {
    return unnamed;
  }

  public void setUnnamed(boolean unnamed) {
    this.unnamed = unnamed;
  }

  /**
   * Called by a transition to pass execution to this node. Execution of this activity is postponed
   * until target links are resolved.
   */
  public void enter(ExecutionContext exeContext) {
    Token token = exeContext.getToken();

    // update the runtime context information
    token.setNode(this);

    // fire the enter-node event for this node
    fireEvent(Event.EVENTTYPE_NODE_ENTER, exeContext);

    // keep track of node entrance, so that a node-log can be generated upon leaving the node
    token.setNodeEnter(new Date());

    // remove the transition references from the runtime context
    exeContext.setTransition(null);
    exeContext.setTransitionSource(null);

    try {
      if (!targets.isEmpty()) {
        setTargetsToken(token);
        if (!areTargetsDetermined(token) || !evaluateJoinCondition(token))
          return;
      }
      execute(exeContext);
    }
    catch (BpelFaultException e) {
      raiseException(e, exeContext);
    }
  }

  private void setTargetsToken(Token token) {
    for (int i = 0, n = targets.size(); i < n; i++) {
      LinkDefinition target = (LinkDefinition) targets.get(i);
      target.getInstance(token).setTargetToken(token);
    }
  }

  private boolean areTargetsDetermined(Token token) {
    for (int i = 0, n = targets.size(); i < n; i++) {
      LinkDefinition target = (LinkDefinition) targets.get(i);
      if (target.getInstance(token).getStatus() == null)
        return false;
    }
    return true;
  }

  /**
   * Checks if the activity is ready to continue its execution. This depends on: a) having
   * determined the status of all the target links. b) lacking of a join condition or having a
   * positive result of its evaluation When this evaluation is negative and suppressJoinFailure=yes,
   * this operation will skip the activity execution and carry on with the remaining process flow
   * before returning the result.
   */
  private boolean evaluateJoinCondition(Token token) {
    if (joinCondition != null) {
      // evaluate explicit join condition
      if (DatatypeUtil.toBoolean(joinCondition.getEvaluator().evaluate(token)))
        return true;
    }
    else {
      // the implicit condition requires the status of at least one incoming link to be positive
      for (int i = 0, n = targets.size(); i < n; i++) {
        LinkDefinition target = (LinkDefinition) targets.get(i);

        Boolean status = target.getInstance(token).getStatus();
        assert status != null : target;

        if (status.booleanValue())
          return true;
      }
    }

    // join failed, throw join failure unless supressed
    if (!suppressJoinFailure())
      throw new BpelFaultException(BpelConstants.FAULT_JOIN_FAILURE);

    // eliminate path
    skip(new ExecutionContext(token));
    return false;
  }

  /**
   * Called by the implementation of this node to continue execution over the default transition. If
   * the activity has source targets their status is solved before resuming the process execution.
   */
  public void leave(ExecutionContext exeContext) {
    if (!sources.isEmpty()) {
      BpelFaultException sourceFault = null;
      Token token = exeContext.getToken();

      for (int i = 0, n = sources.size(); i < n; i++) {
        LinkDefinition source = (LinkDefinition) sources.get(i);
        /*
         * TODO what happens if a transition condition fails? wait for issue 169 to decide whether
         * the rest of the links have to be resolved and which fault has to be thrown
         */
        try {
          source.determineStatus(token);
          // status resolution could have caused a fault or termination
          if (token.hasEnded())
            return;
        }
        catch (BpelFaultException fault) {
          sourceFault = fault;
        }
      }

      if (sourceFault != null)
        throw sourceFault;
    }

    Transition transition = getDefaultLeavingTransition();
    if (transition != null) {
      leave(exeContext, transition);
    }
    else {
      Token token = exeContext.getToken();

      Node node = token.getNode();
      if (node == null)
        token.setNode(this);

      // fire the leave-node event for this node
      fireEvent(Event.EVENTTYPE_NODE_LEAVE, exeContext);
      // complete the enclosing scope instance
      Scope.getInstance(token).completed();
    }
  }

  /**
   * is the {@link CompositeActivity} or {@link ProcessDefinition} in which this activity is
   * contained.
   */
  public GraphElement getParent() {
    return compositeActivity;
  }

  public ProcessDefinition getProcessDefinition() {
    return compositeActivity.getProcessDefinition();
  }

  // behaviour methods
  // /////////////////////////////////////////////////////////////////////////////

  /**
   * Called by the Runtime on an activity when its enclosing scope is forced to terminate. Concrete
   * activities override the terminate method to perform some logic upon termination.
   */
  public void terminate(ExecutionContext exeContext) {
  }

  /**
   * Called when an activity must be skipped. If the activity has source links, they are set to a
   * negative status before carrying on with the process execution.
   */
  public void skip(ExecutionContext context) {
    eliminatePath(context.getToken());
    leave(context, getDefaultLeavingTransition());
  }

  public void eliminatePath(Token token) {
    if (!targets.isEmpty()) {
      setTargetsToken(token);
      if (!areTargetsDetermined(token))
        return;
    }

    if (!sources.isEmpty())
      setNegativeSourcesStatus(token);
  }

  private void setNegativeSourcesStatus(Token token) {
    for (int i = 0, n = sources.size(); i < n; i++) {
      LinkDefinition source = (LinkDefinition) sources.get(i);
      source.getInstance(token).statusDetermined(false);
    }
  }

  public void targetDetermined(Token token) {
    // ignore the notification until every link is determined
    if (!areTargetsDetermined(token))
      return;

    try {
      if (!equals(token.getNode())) {
        // token left: dealing with a path previously eliminated by a structure
        setNegativeSourcesStatus(token);
      }
      else if (evaluateJoinCondition(token))
        execute(new ExecutionContext(token));
    }
    catch (BpelFaultException e) {
      raiseException(e, new ExecutionContext(token));
    }
  }

  protected boolean suppressJoinFailure() {
    return suppressJoinFailure != null ? suppressJoinFailure.booleanValue()
        : getCompositeActivity().suppressJoinFailure();
  }

  // standard attributes and elements
  // /////////////////////////////////////////////////////////////////////////////

  public JoinCondition getJoinCondition() {
    return joinCondition;
  }

  public void setJoinCondition(JoinCondition joinCondition) {
    this.joinCondition = joinCondition;
  }

  public Boolean getSuppressJoinFailure() {
    return suppressJoinFailure;
  }

  public void setSuppressJoinFailure(Boolean suppressJoinFailure) {
    this.suppressJoinFailure = suppressJoinFailure;
  }

  /**
   * Retrieves a target link by name.
   */
  public LinkDefinition getTarget(String name) {
    if (!targets.isEmpty()) {
      for (int i = 0, n = targets.size(); i < n; i++) {
        LinkDefinition target = (LinkDefinition) targets.get(i);
        if (name.equals(target.getName()))
          return target;
      }
    }
    return null;
  }

  public List getTargets() {
    return targets;
  }

  /**
   * Adds a bidirectional relation between this activity and the given target link.
   * @throws IllegalArgumentException if link is null
   */
  public void addTarget(LinkDefinition link) {
    if (link == null)
      throw new IllegalArgumentException("link is null");

    targets.add(link);
    link.setTarget(this);
  }

  /**
   * Removes the bidirectional relation between this activity and the given target link.
   * @throws IllegalArgumentException if link is null
   */
  public void removeTarget(LinkDefinition link) {
    if (link == null)
      throw new IllegalArgumentException("link is null");

    if (targets.remove(link))
      link.setTarget(null);
  }

  /**
   * Retrieves a source link by name.
   */
  public LinkDefinition getSource(String name) {
    if (!sources.isEmpty()) {
      for (int i = 0, n = sources.size(); i < n; i++) {
        LinkDefinition source = (LinkDefinition) sources.get(i);
        if (name.equals(source.getName()))
          return source;
      }
    }
    return null;
  }

  public List getSources() {
    return sources;
  }

  /**
   * Adds a bidirectional relation between this activity and the given source link.
   * @throws IllegalArgumentException if link is null
   */
  public void addSource(LinkDefinition link) {
    if (link == null)
      throw new IllegalArgumentException("link is null");

    sources.add(link);
    link.setSource(this);
  }

  /**
   * Removes the bidirectional relation between this activity and the given source link.
   * @throws IllegalArgumentException if link is null
   */
  public void removeSource(LinkDefinition link) {
    if (link == null)
      throw new IllegalArgumentException("link is null");

    if (sources.remove(link))
      link.setSource(null);
  }

  public CompositeActivity getCompositeActivity() {
    return compositeActivity;
  }

  void setCompositeActivity(CompositeActivity compositeActivity) {
    this.compositeActivity = compositeActivity;
  }

  // utility methods
  // /////////////////////////////////////////////////////////////////////////////

  public void detachFromParent() {
    if (compositeActivity != null) {
      compositeActivity.removeNode(this);
      compositeActivity = null;
    }
  }

  public BpelProcessDefinition getBpelProcessDefinition() {
    return (BpelProcessDefinition) getProcessDefinition();
  }

  public Scope getScope() {
    // easy way out: composite activity is not a scope
    if (!compositeActivity.isScope())
      return compositeActivity.getScope();

    // check whether composite activity has the proper type already
    if (compositeActivity instanceof Scope)
      return (Scope) compositeActivity;

    // reacquire proxy with the proper type
    Session hbSession = JbpmContext.getCurrentJbpmContext().getSession();
    Scope scope = (Scope) hbSession.load(Scope.class, new Long(compositeActivity.getId()));

    // update composite activity reference
    compositeActivity = scope;

    return scope;
  }

  /**
   * Mirrors the {@link #getDefaultLeavingTransition()} method to ease manipulation of activities
   * with a single arriving transition. Notice that every standard activity in a BPEL process has
   * one arriving and one leaving transition because links are implemented as context variables.
   */
  public Transition getDefaultArrivingTransition() {
    return (Transition) getArrivingTransitions().iterator().next();
  }

  /**
   * An activity is initial if there is no basic activity that logically precedes it in the behavior
   * of the process.
   */
  public boolean isInitial() {
    return (compositeActivity == null || compositeActivity.isInitial()
        && compositeActivity.isChildInitial(this))
        && getTargets().isEmpty();
  }

  /**
   * Connects this activity to the given activity by creating an implicit transition between them.
   */
  public void connect(Activity activity) {
    // create an implicit transition
    Transition transition = new Transition();
    transition.setName(getName() + '-' + activity.getName());
    transition.setProcessDefinition(processDefinition);

    // add transition to edge nodes
    addLeavingTransition(transition);
    activity.addArrivingTransition(transition);
  }

  public boolean isConnected(Activity activity) {
    return findTransition(activity) != null;
  }

  /**
   * Disconnects this activity from the given activity by removing the implicit transition between
   * them.
   */
  public void disconnect(Activity activity) {
    Transition transition = findTransition(activity);
    if (transition != null) {
      removeLeavingTransition(transition);
      activity.removeArrivingTransition(transition);
    }
  }

  private Transition findTransition(Activity activity) {
    List leavingTransitions = getLeavingTransitions();
    if (leavingTransitions != null) {

      Set arrivingTransitions = activity.getArrivingTransitions();
      if (arrivingTransitions != null) {

        for (int i = 0, n = leavingTransitions.size(); i < n; i++) {
          Transition transition = (Transition) leavingTransitions.get(i);
          if (arrivingTransitions.contains(transition))
            return transition;
        }
      }
    }
    return null;
  }
}