/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.scheduler.ejbtimer;

import java.rmi.RemoteException;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.ejb.TimedObject;
import javax.ejb.Timer;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.StaleStateException;
import org.hibernate.exception.LockAcquisitionException;

import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmContext;
import org.jbpm.bpel.graph.exe.BpelFaultException;
import org.jbpm.db.JobSession;
import org.jbpm.graph.exe.ExecutionContext;
import org.jbpm.graph.exe.Token;
import org.jbpm.job.Job;

/**
 * @author Alejandro Guizar
 * @version $Revision: 1.4 $ $Date: 2007/08/08 10:50:48 $
 */
public class SchedulerControlBean implements SessionBean, TimedObject {

  private SessionContext sessionContext;
  private JbpmConfiguration jbpmConfiguration;

  /** JNDI name for the jBPM configuration resource. */
  public static final String JBPM_CONFIG_RESOURCE_NAME = "java:comp/env/JbpmCfgResource";

  private static final long serialVersionUID = 1L;
  private static final Log log = LogFactory.getLog(SchedulerControlBean.class);

  public void createTimer(Date dueDate, TimerInfo timerInfo) {
    Timer ejbTimer = sessionContext.getTimerService().createTimer(dueDate, timerInfo);
    log.debug("created ejb timer: " + ejbTimer);
  }

  public void cancelTimers(String timerName, long tokenId) {
    for (Iterator i = sessionContext.getTimerService().getTimers().iterator(); i.hasNext();) {
      Timer ejbTimer = (Timer) i.next();
      TimerInfo timerInfo = (TimerInfo) ejbTimer.getInfo();
      if (timerInfo.matches(timerName, tokenId)) {
        ejbTimer.cancel();
        log.debug("canceled ejb timer: " + ejbTimer);
      }
    }
  }

  public void cancelTimers(long processInstanceId) {
    for (Iterator i = sessionContext.getTimerService().getTimers().iterator(); i.hasNext();) {
      Timer ejbTimer = (Timer) i.next();
      TimerInfo timerInfo = (TimerInfo) ejbTimer.getInfo();
      if (timerInfo.matches(processInstanceId)) {
        ejbTimer.cancel();
        log.debug("canceled ejb timer: " + ejbTimer);
      }
    }
  }

  public void ejbTimeout(Timer ejbTimer) {
    TimerInfo timerInfo = (TimerInfo) ejbTimer.getInfo();
    try {
      log.debug("ejb timeout for: " + ejbTimer);
      executeTimer(timerInfo);
    }
    catch (RuntimeException e) {
      if (isRecoverable(e)) {
        log.warn("timer execution failed due to recoverable exception, attempting reschedule");
        try {
          createTimer(new Date(), timerInfo);
        }
        catch (RuntimeException re) {
          log.error("timer reschedule failed, giving up", re);
        }
      }
      else
        log.error("timer execution failed due to non-recoverable exception, giving up", e);
    }
  }

  private static boolean isRecoverable(RuntimeException exception) {
    for (Throwable throwable = exception; throwable != null; throwable = throwable.getCause()) {
      if (throwable instanceof StaleStateException || throwable instanceof LockAcquisitionException)
        return true;
    }
    return false;
  }

  private void executeTimer(TimerInfo timerInfo) {
    JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
    try {
      JobSession jobSession = jbpmContext.getJobSession();

      long timerId = timerInfo.getTimerId();
      Job jbpmTimer = jobSession.getJob(timerId);

      if (jbpmTimer == null) {
        log.debug("timer was deleted, not executing it: " + timerId);
        return;
      }

      boolean deleteTimer;
      try {
        deleteTimer = jbpmTimer.execute(jbpmContext);
      }
      catch (BpelFaultException e) {
        log.debug("timer execution caused a fault", e);
        Token token = jbpmTimer.getToken();
        token.getNode().raiseException(e, new ExecutionContext(token));
        deleteTimer = true;
      }
      catch (RuntimeException e) {
        // do not threat runtime exceptions as retriable
        throw e;
      }
      catch (Exception e) {
        // exception
        jbpmTimer.setException(e.toString());
        // retries
        int retries = jbpmTimer.getRetries();
        if (retries > 0) {
          jbpmTimer.setRetries(retries - 1);
          deleteTimer = false;
        }
        else
          deleteTimer = true;
      }

      if (deleteTimer)
        jobSession.deleteJob(jbpmTimer);
      else
        jbpmContext.getServices().getSchedulerService().createTimer((org.jbpm.job.Timer) jbpmTimer);
    }
    catch (RuntimeException e) {
      jbpmContext.setRollbackOnly();
      throw e;
    }
    finally {
      jbpmContext.close();
    }
  }

  private static Object lookup(String name) throws NamingException {
    InitialContext initialContext = new InitialContext();
    try {
      return initialContext.lookup(name);
    }
    finally {
      initialContext.close();
    }
  }

  public void setSessionContext(SessionContext sessionContext) throws EJBException, RemoteException {
    this.sessionContext = sessionContext;
  }

  public void ejbCreate() {
    try {
      String configurationName = (String) lookup(JBPM_CONFIG_RESOURCE_NAME);
      jbpmConfiguration = JbpmConfiguration.getInstance(configurationName);
    }
    catch (NamingException e) {
      jbpmConfiguration = JbpmConfiguration.getInstance();
    }
  }

  public void ejbActivate() throws EJBException, RemoteException {
  }

  public void ejbPassivate() throws EJBException, RemoteException {
  }

  public void ejbRemove() throws EJBException, RemoteException {
    sessionContext = null;
  }
}
