/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.admin.console.transform;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.Lob;
import javax.persistence.OneToMany;
import javax.persistence.OrderBy;
import javax.persistence.Transient;

import org.hibernate.annotations.GenericGenerator;
import org.hibernate.validator.NotNull;
import org.jboss.seam.annotations.Name;

/**
 * Transformation Resource Specification.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
@Entity
@Name("transResourceSpec")
public class TransformationResourceSpec implements Serializable {

	private static final long serialVersionUID = 1L;
	private String id;
	private String name;
	private String description;
	private String path;
	private String selector;
	private String namespace;
	private String reference;
	private List<TransformationResourceParameterSpec> parameters;
	
	/**
	 * @return Returns the id.
	 */
	@Id @GeneratedValue(generator="system-uuid")
	@GenericGenerator(name="system-uuid", strategy = "uuid")
	public String getId() {
		return id;
	}
	/**
	 * @param id The id to set.
	 */
	public void setId(String id) {
		this.id = id;
	}
	/**
	 * @return Returns the namespace.
	 */
	public String getNamespace() {
		return namespace;
	}
	/**
	 * @param namespace The namespace to set.
	 */
	public void setNamespace(String namespace) {
		this.namespace = namespace;
	}
	/**
	 * @return Returns the parameters.
	 */
	@OneToMany(mappedBy="resourceSpec")
	@OrderBy("value") // to force the unset and preset specs to be grouped seperately
	public List<TransformationResourceParameterSpec> getParameters() {
		return parameters;
	}
	/**
	 * @param parameters The parameters to set.
	 */
	public void setParameters(List<TransformationResourceParameterSpec> parameters) {
		this.parameters = parameters;
	}
	/**
	 * @return Returns the path.
	 */
	public String getPath() {
		return path;
	}
	/**
	 * @param path The path to set.
	 */
	public void setPath(String path) {
		this.path = path;
	}
	/**
	 * @return Returns the selector.
	 */
	public String getSelector() {
		return selector;
	}
	/**
	 * @param selector The selector to set.
	 */
	public void setSelector(String selector) {
		this.selector = selector;
	}
	/**
	 * @return Returns the description.
	 */
	@Lob
	public String getDescription() {
		return description;
	}
	/**
	 * @param description The description to set.
	 */
	public void setDescription(String description) {
		this.description = description;
	}
	@NotNull
	@Transient
	public String getDescriptionAsString() {
		if(description == null) {
			return null;
		}
		return new String(description);
	}
	public void setDescriptionAsString(String description) {
		this.description = description;
	}
	/**
	 * @return Returns the name.
	 */
	public String getName() {
		return name;
	}
	/**
	 * @param name The name to set.
	 */
	public void setName(String name) {
		this.name = name;
	}
	/**
	 * @return Returns the reference.
	 */
	public String getReference() {
		return reference;
	}
	/**
	 * @param reference The reference to set.
	 */
	public void setReference(String reference) {
		this.reference = reference;
	}
	
	/**
	 * Construct a {@link TransformationResource} instance from this spec.
	 * @return The {@link TransformationResource} instance.
	 */
	@Transient
	public TransformationResource toTransformationResource() {
		TransformationResource transResource = new TransformationResource();
		
		transResource.setName(getName());
		transResource.setPath(getPath());
		transResource.setSelector(getSelector());
		transResource.setNamespace(getNamespace());

		// Copy over the preset resource execution parameters...
		List parameterList = new ArrayList<TransformationResourceParameter>();
		transResource.setParameters(parameterList);
		for(TransformationResourceParameterSpec spec : getParameters()) {
			if(spec.isPreset()) {
				parameterList.add(spec.toTransformationResourceParameter());
			}
		}
		
		
		return transResource;
	}	
}
