/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.admin.console.transform;

import java.util.List;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;

/**
 * Faces converter for a list of transformation resource.
 * <p/>
 * Generates a Smooks XML config fragment.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class SmooksResourceList {
	
	public static String getList() {
		try {
			InitialContext context = new InitialContext();
			EntityManagerFactory entityManagerFactory;
			EntityManager entityManager = null;
			
			// Lookup the EntityManager...
			entityManagerFactory = (EntityManagerFactory) context.lookup("java:/jboss-esb-consoleEntityManagerFactory");
			entityManager = entityManagerFactory.createEntityManager();
			
			try {
				return getList(entityManager);
			} finally {
				if(entityManager != null) {
					entityManager.close();
				}
			}
		} catch (NamingException e) {
			throw new IllegalStateException("Cannot generate Transformation Configurations. Unable to access persistence context.", e);
		}
	}
	
	public static String getList(EntityManager entityManager) {
		List<TransformationResource> resList;
		
		resList = entityManager.createQuery("from TransformationResource").getResultList();

		return getList(resList);
	}

	protected static String getList(List<TransformationResource> resList) {
		StringBuffer stringBuffer = new StringBuffer();
		
		stringBuffer.append("<?xml version=\"1.0\"?>\n");
		stringBuffer.append("<!DOCTYPE smooks-resource-list PUBLIC \"-//MILYN//DTD SMOOKS 1.0//EN\" \"http://www.milyn.org/dtd/smooksres-list-1.0.dtd\">\n");
		stringBuffer.append("<smooks-resource-list>\n");
		
		if(resList != null && !resList.isEmpty()) {
			for(int i = 0; i < resList.size(); i++) {
				TransformationResource resource = (TransformationResource) resList.get(i);
				
				appendResource(resource, stringBuffer);
			}
		} else {
			stringBuffer.append("<!-- No configurations found. -->");
		}

		stringBuffer.append("\n</smooks-resource-list>");
		
		return stringBuffer.toString();
	}

	private static void appendResource(TransformationResource resource, StringBuffer stringBuffer) {
		
		stringBuffer.append("\n\t<smooks-resource ");
		if(resource.getSelector() != null && !resource.getSelector().trim().equals("")) {
			stringBuffer.append("selector=\"").append(resource.getSelector()).append("\" ");
		}
		if(resource.getUseragent() != null && !resource.getUseragent().trim().equals("")) {
			stringBuffer.append("useragent=\"").append(resource.getUseragent()).append("\" ");
		}
		if(resource.getNamespace() != null && !resource.getNamespace().trim().equals("")) {
			stringBuffer.append("namespace=\"").append(resource.getNamespace()).append("\" ");
		}
		if(resource.getPath() != null && !resource.getPath().trim().equals("")) {
			stringBuffer.append("path=\"").append(resource.getPath()).append("\" ");
		}
		stringBuffer.append(">");

		List<TransformationResourceParameter> parameters = resource.getParameters();
		if(parameters != null && !parameters.isEmpty()) {
			appendParameters(resource, parameters, stringBuffer);
		}
	
		stringBuffer.append("\n\t</smooks-resource>");
	}

	private static void appendParameters(TransformationResource resource, List<TransformationResourceParameter> parameters, StringBuffer stringBuffer) {
		TransformationResourceParameter nameParameter = 
			new TransformationResourceParameter(SmooksUtils.RESOURCE_NAME_PARAM, null, resource.getName());
		
		// Append the "Name" parameter.  This is not a "real" parameter.  It's just how we get over the fact that the SmooksResourceConfiguration
		// class doesn't have "name" attribute like the TransformationResource class.  So, we import and export
		// as a resource parameter.
		appendParameter(stringBuffer, nameParameter);
		for(TransformationResourceParameter param : parameters) {
			appendParameter(stringBuffer, param);
		}
	}

	private static void appendParameter(StringBuffer stringBuffer, TransformationResourceParameter param) {
		stringBuffer.append("\n\t\t<param name=\"").append(param.getName()).append("\"");
		if(param.getType() != null && !param.getType().trim().equals("")) {
			stringBuffer.append(" type=\"").append(param.getType()).append("\"");
		}
		stringBuffer.append(">");
		
		stringBuffer.append("<![CDATA[");
		stringBuffer.append(param.getValueAsString());
		stringBuffer.append("]]>");
		stringBuffer.append("</param>");
	}	
}
