/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.admin.console.transform;

import static org.jboss.seam.ScopeType.CONVERSATION;

import java.io.Serializable;
import java.util.List;

import javax.persistence.EntityManager;

import org.jboss.seam.annotations.Begin;
import org.jboss.seam.annotations.End;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Out;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.annotations.datamodel.DataModel;
import org.jboss.seam.annotations.datamodel.DataModelSelectionIndex;
import org.jboss.seam.core.FacesMessages;
import org.jboss.soa.esb.admin.console.SeamUtils;
import org.jboss.soa.esb.admin.console.SeamViewIdStack;
import org.jboss.soa.esb.admin.console.UpdateNotifier;
import org.jboss.soa.esb.admin.console.exchange.MessageExchange;
import org.jboss.soa.esb.admin.console.exchange.SelectMessageExchangeAction;

/**
 * Add Transformation Resource action handler.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
@Name("newResource")
@Scope(CONVERSATION)
public class NewResource implements Serializable {

	private static final long serialVersionUID = 1L;

	@In(create=true)
	private EntityManager entityManager;

	@In(required=false)
	private MessageExchange messageExchange;

	@In(create=true)
	private UpdateNotifier updateNotifier;
	
	@DataModel 
	private List<TransformationResourceSpec> configurationSpecs;
	@DataModelSelectionIndex("configurationSpecs")
	int configurationSpecIndex = 0;
	
	@In(required=false) @Out(required=false)
	private TransformationResourceSpec selectedConfigSpec;
	
	@DataModel 
	private List<TransformationResourceParameterSpec> selectedSpecParams;
	@DataModelSelectionIndex("selectedSpecParams")
	int selectedSpecParamIndex = 0;
	
	@In(create=true) @Out
	private TransformationResourceParameterSpec transResourceParameterSpec;
	
	@In(create=true) @Out
	private TransformationResource transResource;
	
	@In(create=true) @Out
	private TransformationResourceParameter transResourceParameter;
	
	@DataModel 
	private List<TransformationResourceParameter> parameterList;

	@Begin(join=true)
	public void initialise() {
		configurationSpecs = entityManager.createQuery("from TransformationResourceSpec order by name").getResultList();
	}
	
	@Begin(join=true)
	public String selectResourceSpec() {
		initialise();
		
		if(!SelectMessageExchangeAction.isMessageExchangeSelected()) {
			SeamViewIdStack.getInstance().pushViewId("new-trans-resource-select-spec");
			return "select-message-exchange";
		}
		
		return "new-trans-resource-select-spec";
	}
	
	public String captureResourceDetailsConfiguration() {
		selectedConfigSpec = configurationSpecs.get(configurationSpecIndex);
		selectedSpecParams = selectedConfigSpec.getParameters();

		// Create the new resource from the selected spec and set the useragent...
		transResource = selectedConfigSpec.toTransformationResource();
		parameterList = transResource.getParameters();
		transResource.setUseragent(messageExchange.getExchangeUseragentString());
		
		return "new-trans-resource-create";
	}
	
	public String selectParameterSpec() {
		transResourceParameterSpec = selectedSpecParams.get(selectedSpecParamIndex);
		transResourceParameter.setName(transResourceParameterSpec.getName());
		transResourceParameter.setType(transResourceParameterSpec.getType());
		
		return "add-resource-parameter";
	}	
	
	public String addParameter() {
		if(!parameterList.contains(transResourceParameter)) {
			// Protected against the parameter being added more than once.  I've seen this
			// happen occassionaly.
			parameterList.add(transResourceParameter);
		}
		transResourceParameter = new TransformationResourceParameter();
		
		return "new-trans-resource-create";
	}
	
	public String targetConfig() {
		// Verify that all required parameters have been specified...
		if(!assertAllRequiredParamsSet()) {
			return null;
		}

		return "new-trans-resource-target";
	}
	
	public String cancelAddParameter() {
		return "new-trans-resource-create";
	}
	
	@End
	public String save() {
		try {
			// Persist...
			persistNewResource(transResource, entityManager);
			updateNotifier.sendNotification();
			
			return "home";
		} catch(Throwable thrown) {
			SeamUtils.printContexts();
			return "error";
		}
	}

	public static void persistNewResource(TransformationResource transResource, EntityManager esbDatabase) {
		esbDatabase.persist(transResource);
		for(TransformationResourceParameter parameter : transResource.getParameters()) {
			parameter.setTransResource(transResource);
			esbDatabase.persist(parameter);
		}
	}

	/**
	 * Assert that all parameters that have been defined as "required" on the resource spec
	 * are set.
	 */
	private boolean assertAllRequiredParamsSet() {
		for(TransformationResourceParameterSpec paramSpec : selectedSpecParams) {
			if(paramSpec.isRequired()) {
				if(!TransformationResource.isParamaterSet(paramSpec.getName(), transResource)) {
					FacesMessages.instance().add("Required execution parameter '" + paramSpec.getName() + "' not set.");
					return false;
				}
			}
		}
		
		return true;
	}
	
	@End
	public String home() {
		return "home";
	}
}
