/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.monitoring.server;

import java.awt.Color;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;

import org.apache.log4j.Logger;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartUtilities;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.DateAxis;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.time.Minute;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.ui.RectangleInsets;


/**
 * TimeSeriesChartProducer is a helper class which produces a chart
 * based on the time series stored.
 * 
 * @author <a href="mailto:tcunning@redhat.com">Tom Cunningham</a>
 * @since Version 4.2
 */
public class TimeSeriesChartProducer {
	private static final Logger logger = Logger.getLogger(TimeSeriesChartProducer.class);
	private TimeSeries ts;
	private String title;
	
	/**
	 * Constructor
	 * @param title title
	 * @param klass time period class
	 */
	public TimeSeriesChartProducer(String title, Class klass) {
		ts = new TimeSeries(title, klass);	
		this.title = title;
	}
	
	/**
	 * Add minute data
	 * @param min minute
	 * @param num data
	 */
	public void addMinute(Minute min, Number num) {
		ts.addOrUpdate(min, num);
	}
	
	@SuppressWarnings("deprecation")
	public JFreeChart createChart(String chartTitle) {
		TimeSeriesCollection dataset = new TimeSeriesCollection();
		dataset.addSeries(ts);
		dataset.setDomainIsPointsInTime(true);
		
		JFreeChart jfc = ChartFactory.createTimeSeriesChart(
				chartTitle,
				"Date", title,
				dataset, true,
				true, false
			);
		jfc.setBackgroundPaint(Color.white);
		XYPlot plot = (XYPlot) jfc.getPlot();
		plot.setBackgroundPaint(Color.lightGray);
		plot.setDomainGridlinePaint(Color.white);
		plot.setRangeGridlinePaint(Color.white);
		plot.setAxisOffset(new RectangleInsets(5.0, 5.0, 5.0, 5.0));
		plot.setDomainCrosshairVisible(true);
		plot.setRangeCrosshairVisible(true);
		
        XYItemRenderer r = plot.getRenderer();
        if (r instanceof XYLineAndShapeRenderer) {
            XYLineAndShapeRenderer renderer = (XYLineAndShapeRenderer) r;
            renderer.setBaseShapesVisible(true);
            renderer.setBaseShapesFilled(true);
        }

        DateAxis axis = (DateAxis) plot.getDomainAxis();
        axis.setDateFormatOverride(new SimpleDateFormat("MM-dd HH:mm:ss"));
		
		return jfc;
	}
	
	/**
	 * Save the chart to a unique filename by session.
	 * 
	 * @param chartTitle title of chart
	 * @param outFile file to write to
	 */
	public void saveChart(String chartTitle, File outFile) {
		JFreeChart chart = createChart(chartTitle);
		try {
			ChartUtilities.saveChartAsJPEG(outFile, chart, 500, 300);
		} catch (IOException ioe) {
			logger.error("", ioe);
		}
	}
}
