/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.monitoring.server;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import org.apache.log4j.Logger;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SQLQuery;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.soa.esb.monitoring.pojo.*;

/**
 * StatisticsHelper is a helper class which contains methods used in displaying
 * stored JMX attribute information.  
 * 
 * @author <a href="mailto:tcunning@redhat.com">Tom Cunningham</a>
 * @since Version 4.2
 */
public class StatisticsHelper {
	private static final Logger logger = Logger.getLogger(StatisticsHelper.class);
	
	public static final int COUNT_ATTRIBUTE = 0;
	public static final int BYTES_ATTRIBUTE = 1;
	public static final int TIME_ATTRIBUTE = 2;
	public static final int TEXT_ATTRIBUTE = 3;
	
	/**
	 * Return the string "SELECTED" if arg1 == arg2.   Used in
	 * making sure the correct option of the time period select box 
	 * is selected by default.
	 * 
	 * @param arg1 argument 1
	 * @param arg2 argument 2
	 * @return "SELECTED" if the two arguments are equal
	 */
	public static String getSelected(int arg1, int arg2) {
		if (arg1 == arg2) {
			return "SELECTED";
		}
		return "";
	}
	
	/**
	 * Returns an int representing what type of metric is stored 
	 * (count, bytes, or time) depending on what the name of the attribute
	 * is.
	 * @param attribute attribute string
	 * @return attribute type int
	 */
	public static int getAttributeType(String attribute) {
		int attributeType = -1;
		if (attribute.matches("(?i).*Count")) {
			attributeType = COUNT_ATTRIBUTE;
		} else if (attribute.matches("(?i).*Bytes")) { 
			attributeType = BYTES_ATTRIBUTE;
		} else if (attribute.matches("(?i).*Time")) {
			attributeType = TIME_ATTRIBUTE;
		} else {
			attributeType = TEXT_ATTRIBUTE;
		}
		return attributeType;
	}

	/**
	 * Gets the collection interval from JMX.
	 * @return collection interval in minutes
	 */
	public static int getCollectionInterval() {
		MBeanServer mbeanServer = null;
		ObjectName dataSchedulerName = null;
		Integer pollFreq = -1;
		try {
			mbeanServer = MBeanServerLocator.locateJBoss();
		} catch (IllegalStateException ise) {
			logger.error("", ise);
			return -1;
		}
		
		try {
			dataSchedulerName = new ObjectName("jboss.esb:service=DataFilerScheduler");
		} catch (MalformedObjectNameException e) {
			logger.error("", e);
		} catch (NullPointerException e) {
			logger.error("", e);
		}
		
		try {
			pollFreq = (Integer) mbeanServer.getAttribute(dataSchedulerName, "PollMinuteFrequency");
		} catch (AttributeNotFoundException e) {
			logger.error("", e);
		} catch (InstanceNotFoundException e) {
			logger.error("", e);
		} catch (MBeanException e) {
			logger.error("", e);
		} catch (ReflectionException e) {
			logger.error("", e);
		}
		return pollFreq.intValue();
	}
	
	/**
	 * Query Hibernate for all time attributes.
	 * @param sess hibernate session 
	 * @return list of all time attributes
	 */
	public static List getTimeAttributes(Session sess) {
		String query = "from JMXAttribute jmxa where jmxa.attribute like '%Time'";
		List result = sess.createQuery(query).list();;	
		return result;
	}
	
	/**
	 * Query Hibernate for all count attributes.
	 * @param sess hibernate session
	 * @return list of all count attributes
	 */
	public static List getCountAttributes(Session sess) {
		String query = "from JMXAttribute jmxa where jmxa.attribute like '%Count'";
		List result = sess.createQuery(query).list();
		return result;
	}
		
	/**
	 * Query Hibernate for the list of all attributes.
	 * @param sess hibernate session
	 * @return list of all attributes
	 */
	public static List getAllAttributes(Session sess) {
		String query = "from JMXAttribute";
		List result = sess.createQuery(query).list();	
		return result;
	}
	
	/**
	 * Get the list of distinct objectnames in contained in the stored
	 * JMXAttribute information.
	 * @param sess hibernate session
	 * @return list of object names 
	 */
	public static List getObjectNames(Session sess) {
		String query = "SELECT DISTINCT jmxa.objectname "
			+ "FROM JMXAttribute jmxa";
		List result = sess.createQuery(query).list();
		return result;
	}

	/**
	 * Return all JMXAttribute information for a particular statistic id.
	 * @param sess hibernate session
	 * @param id id of attribute
	 * @return JMX attribute
	 */
	public static JMXAttribute getByStatId(Session sess, Long id) {
		String query = "from JMXAttribute where id = :id";  
		Query q = sess.createQuery(query);
		q.setLong("id", id);
		List result = q.list();
		if (result != null) {
			return (JMXAttribute) result.get(0);
		}
		return null;
	}
	
	/**
	 * Select the distinct server names from the data.
	 * @param sess hibernate session
	 * @return list of distinct server names
	 */
	public static List getServerNames(Session sess) {
		String query = "SELECT DISTINCT jmxd.server "
			+ "FROM JMXData jmxd";
		List result = sess.createQuery(query).list();
		return result;
	}
	
	/**
	 * Return the list of distinct ObjectNames for a specific server
	 * @param sess hibernate session 
	 * @param servername server name
	 * @return list of distinct ObjectNames for a specific server
	 */
	public static List getObjectNamesForServer(Session sess, String servername) {
		String query = "SELECT DISTINCT jmxa.objectname "
			+ "FROM JMXATTRIBUTE jmxa, JMXDATA jmxd "
			+ "WHERE jmxa.statid = jmxd.statid "
			+ "AND jmxd.server = :servername";
		SQLQuery sqlq = sess.createSQLQuery(query);
		sqlq.setString("servername", servername);
		List result = sqlq.list();
		return result;
	}
	
	/**
	 * List of distinct attribute names by server and objectname.
	 * @param sess hibernate session 
	 * @param servername server name
	 * @param objectname object name
	 * @return list of attributes
	 */
	public static List getAttributes(Session sess, String servername, String objectname) {
		String query = "SELECT DISTINCT jmxa.* "
			+ "FROM JMXATTRIBUTE jmxa, JMXDATA jmxd "
			+ "WHERE jmxa.statid = jmxd.statid "
			+ "AND jmxd.server = :servername "
			+ "AND jmxa.objectname = :objectname";
		SQLQuery sqlq = sess.createSQLQuery(query);
		sqlq.setString("objectname", objectname);
		sqlq.addEntity("JMXATTRIBUTE", JMXAttribute.class);
		sqlq.setString("servername", servername);
		List result = sqlq.list();
		return result;
	} 
	
	/**
	 * Returns data for display in the bottom half of the window.
	 * @param sess hibernate session
	 * @param servername server name
	 * @param attributeId attribute id
	 * @param timerange time range
	 * @return list of data to display
	 */
	public static List getData(Session sess, String servername, Integer attributeId,
			Integer timerange) {
		String query = "SELECT data.* from JMXDATA data "
			+ "WHERE data.statid = :attributeid "
			+ "AND data.server = :servername ";
		if (timerange.intValue() != -1) {
			query += "AND data.statdate > :datemin ";
		}
			
			query += "ORDER BY data.statdate";
		SQLQuery q = sess.createSQLQuery(query);
		q.setInteger("attributeid", attributeId);
		q.setString("servername", servername);
		if (timerange.intValue() != -1) {
			Calendar calendar = Calendar.getInstance();
			calendar.setTime(new Date());
			calendar.add(Calendar.MINUTE, (timerange.intValue() * -1));
			q.setTimestamp("datemin", new java.sql.Timestamp(calendar.getTimeInMillis()));
		}
		q.addEntity(JMXData.class);
		List result = q.list();
		return result;
	}
}

