/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.monitoring.server;

import java.util.List;

import org.apache.log4j.Logger;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SQLQuery;

/**
 * StatisticsHelper is a helper class which contains methods used in displaying
 * stored JMX attribute information.  
 * 
 * @author <a href="mailto:tcunning@redhat.com">Tom Cunningham</a>
 * @since Version 4.2
 */
public class OperationsHelper {
	private static final Logger logger = Logger.getLogger(OperationsHelper.class);

	/**
	 * Select the distinct server names from the data.
	 * @param sess hibernate session
	 * @return list of distinct server names
	 */
	public static List getServerNames(Session sess) {
		String query = "SELECT DISTINCT jmxo.servername "
			+ "FROM JMXOperation jmxo";
		List result = sess.createQuery(query).list();
		return result;
	}
	
	/**
	 * Return the list of distinct ObjectNames for a specific server
	 * @param sess hibernate session 
	 * @param servername server name
	 * @return list of distinct ObjectNames for a specific server
	 */
	public static List getObjectNamesForServer(Session sess, String servername) {
		String query = "SELECT DISTINCT jmxo.objectname "
			+ "FROM JMXOperation jmxo "
			+ "WHERE jmxo.servername = :servername";
		logger.debug(query);
		SQLQuery sqlq = sess.createSQLQuery(query);
		sqlq.setString("servername", servername);
		List result = sqlq.list();
		return result;
	}
	
	/**
	 * Return the list of operations for a specific server/objectName pair
	 * @param sess hibernate session 
	 * @param servername server name
	 * @return list of distinct ObjectNames for a specific server
	 */
	public static List getOperations(Session sess, String serverName, String objectName) {
		String query = "FROM JMXOperation jmxo "
			+ "WHERE jmxo.servername = :servername "
			+ "AND jmxo.objectname = :objectname "
			+ "AND jmxo.activeflag = true";
		logger.debug(query);
		Query q = sess.createQuery(query);
		q.setString("servername", serverName);
		q.setString("objectname", objectName);
		List result = q.list();
		return result;
	}
	
	/**
	 * Return the list of invocation results for a specific server
	 * @param sess hibernate session 
	 * @param servername server name
	 * @return list of distinct ObjectNames for a specific server
	 */
	public static List getOperationResults(Session sess, String serverName) {
		String query = "FROM JMXOperationResult jmxr "
			+ "WHERE jmxr.servername = :servername "
			+ "ORDER BY jmxr.statdate desc";
		logger.debug(query);
		Query q = sess.createQuery(query);
		q.setString("servername", serverName);
		List result = q.list();
		return result;
	}
}
