/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.monitoring.server;

import java.util.Date;

import org.apache.log4j.Logger;
import org.jboss.system.ServiceMBeanSupport;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SchedulerFactory;
import org.quartz.SimpleTrigger;
import org.quartz.impl.StdSchedulerFactory;

/**
 * DataFilerScheduler is the MBean implementation of DataFilerSchedulerMBean.
 * It contains a pollMinuteFrequency variable, which when updated, reschedules
 * the quartz DataFilerJob.
 * 
 * @author <a href="mailto:tcunning@redhat.com">Tom Cunningham</a>
 * @since Version 4.2
 */
public class DataFilerScheduler extends ServiceMBeanSupport implements DataFilerSchedulerMBean {
	private static final Logger logger = Logger.getLogger(DataFilerScheduler.class);
	private static final int DEFAULT_POLL_MINUTE_FREQ = 2;
	
	public Scheduler scheduler = null;
	
	private int pollMinuteFrequency = DEFAULT_POLL_MINUTE_FREQ;
	
	public int getPollMinuteFrequency() {
		return pollMinuteFrequency;
	}

	public void create() throws Exception {
		super.create();
		SchedulerFactory schedulerFactory = new StdSchedulerFactory();
		scheduler = schedulerFactory.getScheduler();
	}
	
	public void destroy() {
		super.destroy();
		try {
			scheduler.shutdown();
			scheduler.deleteJob(DataFilerJob.JOB_GROUP, DataFilerJob.JOB_NAME);
		} catch (SchedulerException e) {
			logger.error("", e);
		}
	}
	
	public void setPollMinuteFrequency(int f_pollMinuteFrequency) {
		pollMinuteFrequency = f_pollMinuteFrequency;
		
		long ctime = System.currentTimeMillis();
		
		try {
			scheduler.deleteJob(DataFilerJob.JOB_GROUP, DataFilerJob.JOB_NAME);
			
			JobDetail jobDetail = new JobDetail(DataFilerJob.JOB_GROUP, DataFilerJob.JOB_NAME, DataFilerJob.class);

			final JobDataMap datamap = new JobDataMap() ;
            datamap.put(ClassLoader.class.getName(), Thread.currentThread().getContextClassLoader()) ;
			jobDetail.setJobDataMap(datamap) ;
			
			SimpleTrigger simpleTrigger = new SimpleTrigger(DataFilerJob.JOB_GROUP, DataFilerJob.JOB_NAME);
			simpleTrigger.setStartTime(new Date(ctime + 10000));
			simpleTrigger.setRepeatInterval(1000 * 60 * pollMinuteFrequency);
			simpleTrigger.setRepeatCount(-1);
			
			scheduler.scheduleJob(jobDetail, simpleTrigger);
			scheduler.start();
		} catch (SchedulerException se) {
			logger.error("", se);
		} catch (Exception e) {
			logger.error("", e);
		}
	}
	
	
	
}
