package org.jboss.configapp;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.jboss.system.server.ServerConfig;
import org.jboss.system.server.ServerConfigLocator;

/**
 * Used for locating JBossESB server config files.
 * 
 * @author Derek Adams
 */
public class JBossESBConfigLocator {

	/** Suffix for a standalone JBossESB config file */
	private static final String STANDALONE_CONFIG_SUFFIX = "-esb.xml";

	/** Suffix for a standalone JBossESB config file */
	private static final String PACKAGED_CONFIG_SUFFIX = ".esb";

	/** Suffix for a standalone JBossESB config file */
	private static final String NESTED_CONFIG_PATH = "META-INF/jboss-esb.xml";

	/**
	 * Gets the list of JBossESB files in the deploy directory.
	 * 
	 * @return
	 */
	public static List<File> locateConfigFiles() {
		// List all files in the config directory.
		ServerConfig config = ServerConfigLocator.locate();
		File deployDir = new File(config.getServerHomeDir(), "deploy");
		File[] files = deployDir.listFiles();
		List<File> matches = new ArrayList<File>();
		for (File file : files) {
			if (JBossESBConfigLocator.isConfigFile(file)) {
				matches.add(file);
			}
		}
		return matches;
	}

	/**
	 * Indicates if the given file is a JBossESB config file.
	 * 
	 * @param file
	 * @return
	 */
	public static boolean isConfigFile(File file) {
		return ((file.getName().endsWith(PACKAGED_CONFIG_SUFFIX)) || (file
				.getName().endsWith(STANDALONE_CONFIG_SUFFIX)));
	}

	/**
	 * Indicates if the given file is a *-esb.xml file.
	 * 
	 * @param file
	 * @return
	 */
	public static boolean isStandaloneDeployment(File file) {
		return file.getName().endsWith(STANDALONE_CONFIG_SUFFIX);
	}

	/**
	 * Indicates if the given file is a folder containing an exploded
	 * deployment.
	 * 
	 * @param file
	 * @return
	 */
	public static boolean isExplodedDeployment(File file) {
		return (file.getName().endsWith(PACKAGED_CONFIG_SUFFIX))
				&& (file.isDirectory());
	}

	/**
	 * Indicates if the given file is a zipped archive containing a JBossESB
	 * deployment.
	 * 
	 * @param file
	 * @return
	 */
	public static boolean isZippedDeployment(File file) {
		return (file.getName().endsWith(PACKAGED_CONFIG_SUFFIX))
				&& (!file.isDirectory());
	}

	/**
	 * Get an input stream for the JBossESB config taking into account the
	 * different deployment types.
	 * 
	 * @param file
	 * @return
	 * @throws IOException
	 */
	public static InputStream getConfigXmlInputStream(File file)
			throws IOException {
		if (isStandaloneDeployment(file)) {
			if (file.exists()) {
				return new FileInputStream(file);
			}
		} else if (isExplodedDeployment(file)) {
			File configFile = new File(file, NESTED_CONFIG_PATH);
			if (configFile.exists()) {
				return new FileInputStream(configFile);
			}
		} else if (isZippedDeployment(file)) {
			ZipFile zip = new ZipFile(file, ZipFile.OPEN_READ);
			ZipEntry entry = zip.getEntry(NESTED_CONFIG_PATH);
			if (entry != null) {
				return zip.getInputStream(entry);
			}
		}
		return null;
	}

	/**
	 * Get an input stream for the JBossESB config taking into account the
	 * different deployment types.
	 * 
	 * @param file
	 * @return
	 * @throws IOException
	 */
	public static OutputStream getConfigXmlOutputStream(File file)
			throws IOException {
		if (isStandaloneDeployment(file)) {
			if (file.exists()) {
				return new FileOutputStream(file);
			}
		} else if (isExplodedDeployment(file)) {
			File configFile = new File(file, NESTED_CONFIG_PATH);
			if (configFile.exists()) {
				return new FileOutputStream(configFile);
			}
		}
		// NOTE: Writing to zipped archives not supported.
		return null;
	}

	/**
	 * Gets a JBossESB config file by its unique name.
	 * 
	 * @param name
	 * @return
	 */
	public static File getConfigFileByName(String name) {
		List<File> configs = JBossESBConfigLocator.locateConfigFiles();
		for (File file : configs) {
			if (file.getName().equals(name)) {
				return file;
			}
		}
		return null;
	}
}