/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.soap;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.publish.ActionContractPublisher;
import org.jboss.internal.soa.esb.publish.ContractInfo;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ActionDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel.PropertyDocument;
import org.jboss.wsf.spi.deployment.Endpoint;
import org.milyn.Smooks;
import org.milyn.container.standalone.StandaloneExecutionContext;
import org.xml.sax.SAXException;

import javax.servlet.http.HttpServletRequest;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.net.URI;
import java.net.URL;
import java.util.List;

/**
 * Contract publisher for a Webservice endpoint.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class WebserviceContractPublisher implements ActionContractPublisher {

    private static Logger logger = Logger.getLogger(WebserviceContractPublisher.class);    
    private String endpointName;
    private Smooks transformer;

    /**
     * Set the {@link SOAPProcessor} action configuration.
     * @param actionConfig action config.
     * @throws ConfigurationException Bad config.
     */
    public void setActionConfig(ActionDocument.Action actionConfig) throws ConfigurationException {
        List<PropertyDocument.Property> properties = actionConfig.getPropertyList();

        for(PropertyDocument.Property property : properties) {
            if(property.getName().equals(SOAPProcessor.JBOSSWS_ENDPOINT)) {
                endpointName = property.getValue();
                break;
            }
        }

        if(endpointName == null) {
            throw new ConfigurationException("Property '" + SOAPProcessor.JBOSSWS_ENDPOINT + "' not specified.");
        }

        initializeTransform();
    }

    /**
     * Get the contract configuration.
     * <p/>
     * This method impl basically returns the Endpoint WSDL, modified for
     * the supplied EPR and its channel.
     *
     * @param epr Endpoint EPR.
     * @return WSDL Contract.
     */
    public ContractInfo getContractInfo(EPR epr) {
        HttpServletRequest httpServletRequestProxy;

        httpServletRequestProxy = (HttpServletRequest) Proxy.newProxyInstance(HttpServletRequest.class.getClassLoader(),
                                          new Class[] { HttpServletRequest.class },
                                          new HttpServletRequestHandler());

        return getContractInfo(epr, httpServletRequestProxy);
    }

    public ContractInfo getContractInfo(EPR epr, HttpServletRequest servletRequest) {
        Endpoint endpoint = SOAPProcessor.getServiceEndpoint(endpointName);

        if(endpoint != null) {
            String targetServiceCat = servletRequest.getParameter("serviceCat");
            String targetServiceName = servletRequest.getParameter("serviceName");
            String targetProtocol = servletRequest.getParameter("protocol");

            try {
                // Generate the WSDL...
                String wsdl = getEndpointWsdl(endpoint);

                wsdl = updateWsdl(wsdl, epr, targetServiceCat, targetServiceName, targetProtocol);

                return new ContractInfo("text/xml", wsdl);
            } catch (Exception e) {
                logger.error("Failed to load contract information from JBossWS Endpoint '" + endpointName + "'.", e);
                return new ContractInfo("text/plain", "Unavailable: " + e.getMessage());
            }
        } else {
            logger.warn("Requested contract info for unknown webservice endpoint '" + endpointName + "'.");
            return null;
        }
    }

    /**
     * Get the endpoints WSSDL from the JBossWS container.
     * </p>
     * See http://jira.jboss.com/jira/browse/JBWS-1909
     *
     * @param endpoint The endpoint.
     * @return The WSDL.
     * @throws IOException Error reading wsdl.
     */
    private String getEndpointWsdl(Endpoint endpoint) throws IOException {
        URL endpointWsdlAddress = new URL(endpoint.getAddress() + "?wsdl");
        InputStream wsdlStream = endpointWsdlAddress.openStream();

        try {
            return StreamUtils.readStreamString(wsdlStream, "UTF-8");
        } finally {
            wsdlStream.close();
        }
    }

    /**
     * Update the supplied wsdl to take account of the ESB endpoint proxying of the JBossWS
     * invocation, as well as the fact that the transport may be different.
     *
     * @param wsdl WSDL input.
     * @param epr The SOAP endpoint from the ESB perspective.
     * @param targetServiceCat
     *@param targetServiceName
     * @param targetProtocol @return The updated WSDL.
     */
    protected String updateWsdl(String wsdl, EPR epr, String targetServiceCat, String targetServiceName, String targetProtocol) throws SAXException, IOException, ConfigurationException {
        URI endpointURI = URI.create(epr.getAddr().getAddress());
        StringWriter writer = new StringWriter();
        StandaloneExecutionContext execContext = transformer.createExecutionContext();

        execContext.setAttribute(WsdlTransformer.ENDPOINT_URI, endpointURI);
        execContext.setAttribute(WsdlTransformer.TARGET_CAT, targetServiceCat);
        execContext.setAttribute(WsdlTransformer.TARGET_NAME, targetServiceName);
        execContext.setAttribute(WsdlTransformer.TARGET_PROTOCOL, targetProtocol);
        transformer.filter(new StreamSource(new StringReader(wsdl)), new StreamResult(writer), execContext);

        return writer.toString().trim();
    }

    protected void initializeTransform() throws ConfigurationException {
        try {
            transformer = new Smooks("/org/jboss/soa/esb/actions/soap/wsdltrans.xml");
        } catch (IOException e) {
            throw new ConfigurationException("Failed to read the Smooks Transformation config for WSDL.", e);
        } catch (SAXException e) {
            throw new ConfigurationException("Failed to read the Smooks Transformation config for WSDL.", e);
        }
    }

    private class HttpServletRequestHandler implements InvocationHandler {
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            if(method.getName().equals("getRequestURL")) {
                return new StringBuffer("http://www.jboss.org");
            }

            return null;
        }
    }
}
