/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.client.MessageMulticaster;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.mapping.ObjectMapper;
import org.jboss.soa.esb.message.mapping.ObjectMappingException;
import org.jboss.soa.esb.services.registry.Registry;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.RegistryFactory;
import org.jboss.soa.esb.services.routing.MessageRouterException;
import org.jboss.soa.esb.services.routing.cbr.ContentBasedRouterFactory;

/**
 * @author <a
 *         href="mailto:schifest@heuristica.com.ar">schifest@heuristica.com.ar</a>
 * @author kstam@jboss.com
 * @author kevin.conner@jboss.com
 */
public class ContentBasedWiretap extends AbstractActionPipelineProcessor {

    public static final String ROUTE_TO_TAG = "route-to";

    public static final String OBJECT_PATH_TAG = "object-path";

    public static final String OBJECT_PATH = "esb";

    public static final String DEFAULT_CBR_CLASS = "org.jboss.internal.soa.esb.services.routing.cbr.JBossRulesRouter";

    ServiceInvoker dlQueueInvoker;

    public ContentBasedWiretap(ConfigTree config)
            throws ConfigurationException, RegistryException,
            MessageRouterException {
        _config = config;
        checkMyParms();
        _registry = RegistryFactory.getRegistry();
        _cbr = ContentBasedRouterFactory.getRouter(_cbrClass);
        _cbr.setConfigTree(config);
        _mapper = new ObjectMapper(config);
        try {
            dlQueueInvoker = new ServiceInvoker(ServiceInvoker.INTERNAL_SERVICE_CATEGORY, ServiceInvoker.DEAD_LETTER_SERVICE_NAME);
        } catch (MessageDeliverException e) {
            new MessageRouterException(e);
        }
    }

    public void initialise() {
        if (messageMulticaster.getRecipientCount() == 0) { 
            _logger.info("Missing or empty destination list - This action class won't have any effect");
        }
    }

    /**
     * Router the message to one or more destinations, using the
     * ContentBasedRouter to figure out to which destinations it is going to
     * be routed too.
     *
     * @param message
     * @return Message
     * @throws ActionProcessingException
     */
    public Message process(Message message) throws ActionProcessingException 
    {
        try {
            List<Service> outgoingDestinations = executeRules(message);
            if (outgoingDestinations.size()==0) {
                String error = "No rule destination(s) "+ _destinations.keySet() + " were matched, "
                           + ". Please fix your configuration and/or routing rules.";
                _logger.error(error);
                try {
                    _logger.debug("Sending message to the DeadLetterService");
                    dlQueueInvoker.deliverAsync(message);
                    throw new ActionProcessingException(error);
                } catch (MessageDeliverException e) {
                    throw new MessageRouterException("Failed to deliver message to Dead Letter Channel.", e);
                }
            } else {
                routeMessage(message, outgoingDestinations);
            }
        } catch (MessageRouterException e) {
            throw new ActionProcessingException(e);
        }
        return message;
    }
    
    protected List<Service> executeRules(Message message)
        throws MessageRouterException 
    {
        List<Service> outgoingDestinations = new ArrayList<Service>();
        try {
            List<Object> objectList = _mapper.createObjectList(message,
                    _messagePathList);
            List<String> destinations = _cbr.route(_ruleSet, _ruleLanguage,
                    _ruleReload, message, objectList);
            for (String destination : destinations) {
                if (_destinations.containsKey(destination)) {
                    outgoingDestinations.add(_destinations.get(destination));
                } else {
                    throw new MessageRouterException("Destination " + destination + " does not exist your configuration");
                }
            }
        } catch (ObjectMappingException ome) {
            throw new MessageRouterException(ome);
        }
        return outgoingDestinations;
    }

    protected final void routeMessage(Message message, List<Service> outgoingDestinations)
        throws MessageRouterException 
    {  
        try {
            messageMulticaster.sendToSubset(message, outgoingDestinations);
        } catch (RegistryException e) {
            throw new MessageRouterException(e);
        } catch (MessageDeliverException e) {
            throw new MessageRouterException(e);
        }
    }

    /**
     * Reading the piece of configTree specific to the CBR, and setting the
     * configuration.
     *
     * @throws ConfigurationException
     */
    protected void checkMyParms() throws ConfigurationException {
        if (_config.getAttribute(ListenerTagNames.RULE_SET_TAG) == null) {
            _logger.error("Required attribute " + ListenerTagNames.RULE_SET_TAG
                    + " not found in " + _config.getName() + ".");
            throw new ConfigurationException("Required attribute "
                    + ListenerTagNames.RULE_SET_TAG + " not found.");
        } else {
            _ruleSet = _config.getAttribute(ListenerTagNames.RULE_SET_TAG);
            if (_ruleSet == null) {
                throw new ConfigurationException("Required attribute "
                        + ListenerTagNames.RULE_SET_TAG + " not found.");
            }
            _ruleLanguage = _config
                    .getAttribute(ListenerTagNames.RULE_LANGUAGE_TAG);
            String ruleReload = _config
                    .getAttribute(ListenerTagNames.RULE_RELOAD_TAG);
            if (ruleReload != null && "true".equals(ruleReload)) {
                _ruleReload = true;
            }
        }
        if (_config.getAttribute(ListenerTagNames.CBR_CLASS) != null) {
            _cbrClass = _config.getAttribute(ListenerTagNames.CBR_CLASS);
        } else {
            _cbrClass = DEFAULT_CBR_CLASS;
        }

        _destinations = new HashMap<String, Service>();
        ConfigTree[] destList = _config.getChildren(ROUTE_TO_TAG);
        if (destList != null) {
            for (ConfigTree curr : destList) {
                try {
                    String key = curr
                            .getRequiredAttribute(ListenerTagNames.DESTINATION_NAME_TAG);
                    String category = curr.getAttribute(
                            ListenerTagNames.SERVICE_CATEGORY_NAME_TAG, "");
                    String name = curr
                            .getRequiredAttribute(ListenerTagNames.SERVICE_NAME_TAG);
                    Service service = new Service(category, name);
                    _destinations.put(key, service);
                    messageMulticaster.addRecipient(service);
                }
                catch (Exception e) {
                    throw new ConfigurationException(
                            "Problems with destination list", e);
                }
            }
        }
        _messagePathList = new ArrayList<String>();
        ConfigTree[] objectList = _config.getChildren(OBJECT_PATH_TAG);
        if (objectList != null) {
            for (ConfigTree curr : objectList) {
                try {
                    String objectPath = curr.getRequiredAttribute(OBJECT_PATH);
                    _messagePathList.add(objectPath);
                }
                catch (Exception e) {
                    throw new ConfigurationException(
                            "Problems with object path list", e);
                }
            }
        }

    }

    protected ConfigTree _config;

    protected Map<String, Service> _destinations;

    protected MessageMulticaster messageMulticaster = new MessageMulticaster();

    protected String _cbrClass;

    protected String _ruleSet;

    protected String _ruleLanguage;

    protected boolean _ruleReload;

    protected List<String> _messagePathList;

    protected ObjectMapper _mapper;

    protected Registry _registry;

    protected org.jboss.soa.esb.services.routing.cbr.ContentBasedRouter _cbr;

    protected static Logger _logger = Logger
            .getLogger(ContentBasedWiretap.class);

}
