/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.services.jbpm.actionhandlers;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.dom4j.tree.DefaultElement;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.jbpm.Constants;
import org.jboss.soa.esb.services.jbpm.JBpmObjectMapper;
import org.jboss.soa.esb.services.jbpm.actionhandlers.EsbActionHandler;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.exe.ExecutionContext;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.graph.exe.Token;
import org.junit.BeforeClass;
import org.junit.Test;
/**
 * Tests the capabilities of the JBpmObjectMapper.
 * 
 * @author kstam
 *
 */
public class JBpmObjectMapperCallbackTest 
{
    private static String PROCESS_DEF_XML = "JBpmObjectMapperTestProcessDefinition.xml";
	private static Logger logger = Logger.getLogger(JBpmObjectMapperCallbackTest.class);
    private static ProcessInstance processInstance = null;

    @BeforeClass
    public static void setup()
    {
        logger.info("Setting up jBPM");
        //Extract a process definition from the processdefinition.xml file.
        ProcessDefinition processDefinition 
            = ProcessDefinition.parseXmlResource(PROCESS_DEF_XML);
        assertNotNull(processDefinition);
        //Create an instance of the process definition.
        processInstance = new ProcessInstance(processDefinition);
        assertTrue(
                "Instance is in start state", 
                "start".equals(processInstance.getRootToken().getNode().getName()));
    }

    /**
     * Tests if the variableMapping can go roundtrip (via the callbackService code)
     * 
     * @throws Exception
     */
    @Test
    public void esbToJBpmAndBack() throws Exception
    {
        SAXReader reader = new SAXReader();
        Document document = reader.read(this.getClass().getResourceAsStream("/" + PROCESS_DEF_XML));
        Element element = document.getRootElement();
        DefaultElement bpmToEsbVars = (DefaultElement) element.element("start-state").element("transition").element("action").element("bpmToEsbVars");
        DefaultElement esbToBpmVars = (DefaultElement) element.element("start-state").element("transition").element("action").element("esbToBpmVars");
        
        String helloWorldTokenScope  = "Hello world token scope";
        String helloWorldGlobalScope = "Hello world process-instance scope";
        
        Token token = processInstance.getRootToken();
        processInstance.getContextInstance().createVariable("v1", helloWorldTokenScope, token);
        processInstance.getContextInstance().createVariable("g2", helloWorldGlobalScope);
        ExecutionContext executionContext = new ExecutionContext(token);
        
        JBpmObjectMapper mapper = new JBpmObjectMapper();
        Message message = mapper.mapFromJBpmToEsbMessage(bpmToEsbVars, Boolean.FALSE, executionContext);
        
        //Setting up return mapping for the callback service.
        EsbActionHandler esbActionHandler = new EsbActionHandler();
        EPR epr = esbActionHandler.createReplyTo(esbToBpmVars.asXML(), Boolean.FALSE, executionContext);
        String esbToBpmXml = epr.getAddr().getExtensionValue(Constants.ESB_TO_BPM_VARS_TAG);
        
        //Obtaining the VariableMap that is going to be set callback command
        Map<String,Object> variableMap = mapper.mapFromEsbMessageToJBpmMap(message, esbToBpmXml);
        logger.info(variableMap);
        //Let's make sure that what we put in is going to make it roundtrip.
        assertEquals(helloWorldTokenScope, variableMap.get("x1"));
        assertEquals(helloWorldGlobalScope,variableMap.get("x2"));
    }
    
    public static junit.framework.Test suite(){
        return new JUnit4TestAdapter(JBpmObjectMapperCallbackTest.class);
    }

}
