/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.services.jbpm.actionhandlers;
import org.apache.log4j.Logger;
import org.dom4j.tree.DefaultElement;
import org.jboss.internal.soa.esb.util.LRUCache;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.jbpm.JBpmObjectMapper;
import org.jbpm.graph.def.ActionHandler;
import org.jbpm.graph.exe.ExecutionContext;

/**
 * 
 * Sends ESB messages to ESB services from jBPM in an asynchronous way
 * 
 * <li/>esbCategoryName - for ESB registry lookup 
 * <li/>esbServiceName - for ESB registry lookup
 * 
 * @author <a href="mailto:kurt.stam@jboss.com">Kurt Stam</a>
 * 
 */
public class EsbNotifier implements ActionHandler
{
    private static final long serialVersionUID = 1L;
    private transient Logger logger = Logger.getLogger(getClass());
    private static transient LRUCache<String, ServiceInvoker> siCache = new LRUCache<String, ServiceInvoker>(20);
    /**
     * The ESB Service Category Name, specified in the processdefinition.xml
     */
    public String esbCategoryName;
    /**
     * The ESB Service Name, specified in the processdefinition.xml
     */
    public String esbServiceName;
    /**
     * Default setting of the process scope, can be overriden by setting the
     * process-scope attribute on the mapping element.
     */
    public Boolean globalProcessScope;
    /**
     * XML element to specify the mapping of variables from jBPM to ESB.
     */
    public DefaultElement bpmToEsbVars;
    /**
     * Constructs an ESB Message and sends sends to the an ESB Service
     * as defined in the processdefinition.xml.
     */
    public void execute (ExecutionContext executionContext) throws Exception
    {
        if (null == esbCategoryName) {
            throw new ConfigurationException(
                "Service category (esbCategoryName element) must not be null");
        }
        if (null == esbServiceName) {
            throw new ConfigurationException(
                "Service name (esbServiceName element) must not be null");
        }
        JBpmObjectMapper mapper = new JBpmObjectMapper();
        Message message = mapper.mapFromJBpmToEsbMessage(bpmToEsbVars, globalProcessScope, executionContext);
        if (logger.isDebugEnabled()) logger.debug("Created ESB message=" + message);
        getServiceInvoker().deliverAsync(message);
        logger.debug("Message send successfully");
    }
    /**
     * Caches the most recently used ServiceInvokers.
     * 
     * @return a ServiceInvoker for the current esbService and esbCategoryName.
     * @throws MessageDeliverException
     */
    private ServiceInvoker getServiceInvoker() throws MessageDeliverException
    {
        String key = esbCategoryName + esbServiceName;
        if (siCache.containsKey(key)) {
            return siCache.get(key);
        } else {
            ServiceInvoker invoker = new ServiceInvoker(esbCategoryName,  esbServiceName);
            siCache.put(key, invoker);
            return invoker;
        }
    }

}
