/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.ant;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.ConnectException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.taskdefs.MatchingTask;
import org.apache.tools.ant.types.FileSet;

/**
 * ant task for deploying process archives to the deployment servlet.
 * 
 * @author kurt stam
 */
public class DeployProcessToServerTask extends MatchingTask {

    private static final String boundary = "AaB03x";
    
    String serverName = "localhost";
    String serverPort = "8080";
    String serverDeployer = "/jbpm-console/upload";
    String debug = null;
    File process = null;
    List fileSets = new ArrayList();

    public void setServerDeployer(String serverDeployer) {
        this.serverDeployer = serverDeployer;
    }

    public void setServerName(String serverName) {
        this.serverName = serverName;
    }

    public void setServerPort(String serverPort) {
        this.serverPort = serverPort;
    }
    
    public void setProcess(File process) {
        this.process = process;
    }
    
    public void setDebug(String debug) {
        this.debug = debug;
    }

    public void execute() throws BuildException {
        try {
            if (pingServerOK()) {
                //if attribute par is set, deploy the par file
                if (process!=null) {
                    log( "deploying par "+process.getAbsolutePath()+" ..." );
                    FileInputStream in = new FileInputStream(process);
                    ByteArrayOutputStream out = new ByteArrayOutputStream();
                    byte[] buf = new byte[1024];
                    int len;
                    while ((len = in.read(buf)) > 0) {
                        out.write(buf, 0, len);
                    }
                    out.flush();
                    if (debug!=null) {
                        saveParFile("debug.par", out.toByteArray());
                    }
                    deployProcessWithServlet(serverName, serverPort, serverDeployer, out.toByteArray());
                    in.close();
                    log("deployment complete.");
                }
                //if attribute fileSets is set, deploy the fileSets too
                if (fileSets.size()>0) {
                    ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
                    ZipOutputStream zipOutputStream = new ZipOutputStream(byteArrayOutputStream);
    
                    // loop over all files that are specified in the filesets
                    Iterator iter = fileSets.iterator();
                    while (iter.hasNext()) {
                        FileSet fileSet = (FileSet) iter.next();
                        DirectoryScanner dirScanner = fileSet.getDirectoryScanner(getProject());
                        String[] fileSetFiles = dirScanner.getIncludedFiles();
    
                        for (int i = 0; i < fileSetFiles.length; i++) {
                            String fileName = fileSetFiles[i];
                            File file = new File(fileName);
                            if ( !file.isFile() ) {
                                file = new File( dirScanner.getBasedir(), fileName );
                            }
                            // deploy the file, specified in a fileset element
                            log( "adding to process archive "+file+" ..." );
                            addFile(zipOutputStream, file);
                        }
                    }
                    zipOutputStream.close();
                    log( "deploying par ..." );
                    if (debug!=null) {
                        saveParFile("debug.par", byteArrayOutputStream.toByteArray());
                    }
                    deployProcessWithServlet(serverName, serverPort, serverDeployer, byteArrayOutputStream.toByteArray());
                    log("deployment complete.");
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw new BuildException( "couldn't deploy process archives : " + e.getMessage() );
        }
    }

    public void addFileset(FileSet fileSet) {
        this.fileSets.add(fileSet);
    }

    public void deployProcessWithServlet(String serverName, String serverPort, String serverDeployer, byte[] parBytes) throws Exception {
        URL url = new URL("http://" + serverName + ":" + serverPort + serverDeployer);
        URLConnection urlConnection = url.openConnection();
        urlConnection.setDoInput(true);
        urlConnection.setDoOutput(true);
        urlConnection.setUseCaches(false);
        urlConnection.setRequestProperty("Content-Type",
        "multipart/form-data; boundary=AaB03x");
        DataOutputStream dataOutputStream = new DataOutputStream(urlConnection
                .getOutputStream());
        dataOutputStream.writeBytes("--" + boundary + "\r\n");
        dataOutputStream.writeBytes("Content-Disposition: form-data; name=\"definition\"; filename=\"dummy.par\"\r\n");
        dataOutputStream.writeBytes("Content-Type: application/x-zip-compressed\r\n\r\n");

        dataOutputStream.write(parBytes);

        dataOutputStream.writeBytes("\r\n--" + boundary + "--\r\n");
        dataOutputStream.flush();
        dataOutputStream.close();
        InputStream inputStream = urlConnection.getInputStream();
        StringBuffer result = new StringBuffer();
        int read;
        while ((read = inputStream.read()) != -1) {
            result.append((char)read);
        }
    }

    public void addFile(ZipOutputStream zos, File file)
    throws IOException 
    {
        byte[] buff = new byte[256];
        if (!file.exists()) return;
        InputStream is = new FileInputStream(file);
        zos.putNextEntry(new ZipEntry(file.getName()));
        int read;
        while ((read = is.read(buff)) != -1) {
            zos.write(buff, 0, read);
        }
    }
    
    private void saveParFile(String fileName, byte[] parBytes) throws IOException {
        File file = new File(fileName);
        if (!file.exists()) {
            file.createNewFile();
        }
        FileOutputStream fos = new FileOutputStream(file);
        fos.write(parBytes);
        fos.close();
    }
    
    public boolean pingServerOK() {
        URL url = null;
        try {
            url = new URL("http://" + serverName + ":" + serverPort + serverDeployer);
            URLConnection urlConnection = url.openConnection();
            urlConnection.setDoOutput(true);
            InputStream inputStream = urlConnection.getInputStream();
            StringBuffer result = new StringBuffer();
            int read;
            while ((read = inputStream.read()) != -1) {
                result.append((char)read);
            }
            return true;
        }
        catch (ConnectException e) {
            log("The server (" + url + ") could not be reached.");
            return false;
        }
        catch (Exception e) {
            log("An unexpected exception happened while testing the server (" + url + ") connection.");
            return false;
        }
    }
}