/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.persistence.tests;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.sql.Connection;
import java.sql.Statement;
import java.util.Date;
import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import org.apache.log4j.xml.DOMConfigurator;
import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.persistence.format.MessageStoreFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.actions.MessageRedeliverer;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.parameters.ParamRepositoryException;
import org.jboss.soa.esb.persistence.manager.ConnectionManager;
import org.jboss.soa.esb.persistence.manager.ConnectionManagerFactory;
import org.jboss.soa.esb.services.persistence.MessageStore;
import org.jboss.soa.esb.services.persistence.MessageStoreException;
import org.jboss.soa.esb.services.persistence.RedeliverStore;
import org.jboss.soa.esb.testutils.ESBConfigUtil;
import org.jboss.soa.esb.testutils.HsqldbUtil;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.xml.sax.SAXException;

/**
 * Scheduled Redelivery unit test.
 *
 * @author <a href="mailto:kurt.stam@jboss.com">kurt.stam@jboss.com</a>
 */
public class ScheduledRedeliveryUnitTest{
    
    @Test
    public void test_listener_config() throws UnsupportedEncodingException {
        ESBConfigUtil configUtil = new ESBConfigUtil(getClass().getResourceAsStream("config-01.xml"));
        String config = configUtil.getListenerConfig("simple-schedule-listener").toString();
        assertNotNull(config);
        try {
            ConfigTree tree = ConfigTree.fromXml(config);
            String maxDeliveries = tree.getFirstChild("action").getAttribute(MessageRedeliverer.MAX_REDELIVER_ATTR);
            assertEquals(maxDeliveries, "9");
        } catch (SAXException e) {
            fail(e.getMessage());
        }
    }

    @Test
    public void test_simple_schedule_01() {
        
        try {
            int numberOfMessages=20;
            MessageStore store = MessageStoreFactory.getInstance().getMessageStore();
            assertEquals((store != null), true);
            //first lets create an undeliverable message
            Service service = new Service("cat", "service");
            URI uid = createMessages(numberOfMessages, service, store);
            Map<URI, Message> messageMap = store.getAllMessages(RedeliverStore.CLASSIFICATION_RDLVR); 
            assertEquals(messageMap.size(), numberOfMessages);
            System.out.println(new Date());
            runTestConfig("config-01.xml", 5000);
            System.out.println(new Date());
            Map<URI,Message> leftOverMsgs = store.getAllMessages(MessageStore.CLASSIFICATION_RDLVR);
            System.out.println("Number of messages still in RDLVR store: " + leftOverMsgs.size());
            Map<URI,Message> dlqMsgs = store.getAllMessages(MessageStore.CLASSIFICATION_DLQ);
            System.out.println("Number of messages still in DLQ store: " + dlqMsgs.size());
            Message message2 = store.getMessage(uid); 
            assertNull(message2);
        } catch (Exception e) {
            e.printStackTrace();
            fail(e.getMessage());
        }
    }

    private void runTestConfig(String configName, long upTime) throws ParamRepositoryException, ConfigurationException, ManagedLifecycleException, SAXException, InterruptedException {
        ESBConfigUtil configUtil = new ESBConfigUtil(getClass().getResourceAsStream(configName));

        configUtil.startController();
        Thread.sleep(upTime);
        configUtil.stopController();
    }
    
    private URI createMessages(int numberOfMessages, Service service, MessageStore store) throws MessageStoreException
    {
        URI uid=null;
        for (int i=0; i<numberOfMessages; i++) {
            Message msg = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
            assertEquals((msg != null), true);
            msg.getProperties().setProperty(ServiceInvoker.DELIVER_TO, service);
            uid = store.addMessage(msg, MessageStore.CLASSIFICATION_RDLVR);
        }
        return uid;
    }
    
    @BeforeClass
    public static void runBeforeAllTests()
    {
        MockCourierFactory.install();
        MockRegistry.install();
        MockCourier courier1 = new MockCourier(true);
        MockRegistry.register("cat", "service", courier1);
        
        try {
            File testResourceDir = TestEnvironmentUtil.findResourceDirectory("./product/services/jbossesb/src/test/resources/");
            System.out.println("Current dir=" + testResourceDir.getCanonicalPath());
            DOMConfigurator.configure(testResourceDir.getCanonicalPath() + "/log4j.xml");
            File buildDir = TestEnvironmentUtil.findResourceDirectory("./product/services/jbossesb/build/");
            File resourceDir = TestEnvironmentUtil.findResourceDirectory("./product/services/jbossesb/src/main/resources/");
            System.setProperty("org.jboss.soa.esb.propertyFile", "jbossesb-unittest-properties.xml");                    
            if ("org.hsqldb.jdbcDriver".equals(Configuration.getStoreDriver())) {
                HsqldbUtil.startHsqldb(buildDir + "/hsqltestdb", "jbossesb");
                
                //Drop what is there now, if exists. We want to start fresh.                
                String sqlCreateCmd    = TestEnvironmentUtil.readTextFile(new File(resourceDir.getCanonicalPath() + "/message-store-sql/hsqldb/create_database.sql"));
                String sqlDropCmd      = TestEnvironmentUtil.readTextFile(new File(resourceDir.getAbsolutePath() + "/message-store-sql/hsqldb/drop_database.sql"));
                
                ConnectionManager mgr = ConnectionManagerFactory.getConnectionManager();
                mgr.init();
                Connection con = mgr.getConnection();
                Statement stmnt = con.createStatement();
                System.out.println("Dropping the schema if exist");
                stmnt.execute(sqlDropCmd);
                System.out.println("Creating the message store schema");
                stmnt.execute(sqlCreateCmd);
            } else if ("com.mysql.jdbc.Driver".equals(Configuration.getStoreDriver())) {
                
                String sqlCreateCmd    = TestEnvironmentUtil.readTextFile(new File(resourceDir.getCanonicalPath() + "/message-store-sql/mysql/create_database.sql"));
                String sqlDropCmd      = TestEnvironmentUtil.readTextFile(new File(resourceDir.getCanonicalPath() + "/message-store-sql/mysql/drop_database.sql"));
                
                ConnectionManager mgr = ConnectionManagerFactory.getConnectionManager();
                mgr.init();
                Connection con = mgr.getConnection();
                Statement stmnt = con.createStatement();
                System.out.println("Dropping the schema if exist");
                stmnt.execute(sqlDropCmd);
                System.out.println("Creating the message store schema");
                stmnt.execute(sqlCreateCmd);
                
            }
        } catch (Throwable e) {
            e.printStackTrace();
            System.out.println("We should stop testing, since we don't have a db.");
            assertTrue(false);
        }
        
    }

    @AfterClass
    public static void runAfterAllTests ()
    {
        
        try
        {
            if (Configuration.getStoreDriver().equals("org.hsqldb.jdbcDriver"))
                HsqldbUtil.stopHsqldb(Configuration.getStoreUrl(),
                        Configuration.getStoreUser(), Configuration
                                .getStorePwd());
        }
        catch (Exception e)
        { // 
            e.printStackTrace();
        }
        
         MockRegistry.uninstall();
         MockCourierFactory.uninstall();
    }
    
    public static junit.framework.Test suite ()
    {
        return new JUnit4TestAdapter(ScheduledRedeliveryUnitTest.class);
    }
}
