/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author derek.adams@sapience360.com
 */

package org.jboss.internal.soa.esb.persistence.format.jcr;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Properties;

import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;
import javax.naming.Context;
import javax.naming.NamingException;

import org.apache.commons.lang.StringUtils;
import org.apache.jackrabbit.core.TransientRepository;
import org.apache.jackrabbit.core.config.RepositoryConfig;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.helpers.NamingContextException;
import org.jboss.soa.esb.helpers.NamingContextPool;
import org.jboss.system.server.ServerConfig;
import org.jboss.system.server.ServerConfigLocator;

/**
 * Manages connections to a JSR 170 content repository.
 * 
 * @author Derek Adams
 */
public class JCRConnectionManager {

	/** Singleton instance */
	private static JCRConnectionManager instance;

	/** Repository instance */
	private Repository repository;

	/** Repository session */
	private Session session;

	/**
	 * Singleton constructor.
	 */
	private JCRConnectionManager() {
	}

	/**
	 * Get the singleton manager instance.
	 * 
	 * @return
	 */
	public static JCRConnectionManager getInstance() {
		if (instance == null) {
			instance = new JCRConnectionManager();
		}
		return instance;
	}

	/**
	 * Initialize the repository.
	 * 
	 * @param config
	 * @param home
	 * @return
	 * @throws RepositoryException
	 */
	protected Repository initRepository(File config, File home) throws RepositoryException {
		RepositoryConfig rconfig = RepositoryConfig.create(config.getPath(), home.getPath());
		try {
			return new TransientRepository(rconfig);
		} catch (IOException e) {
			throw new RepositoryException(e);
		}
	}

	/**
	 * Get the repository instance.
	 * 
	 * @return
	 */
	protected Repository getRepository() throws RepositoryException {
		if (repository == null) {
			// If JNDI path specified, load the repository from JNDI.
			if (!StringUtils.isEmpty(Configuration.getJcrStoreJNDIPath())) {
				Properties environment = new Properties();
				environment.setProperty(Context.PROVIDER_URL, Configuration.getJndiServerURL());
				environment.setProperty(Context.INITIAL_CONTEXT_FACTORY,
						Configuration.getJndiServerContextFactory());
				environment.setProperty(Context.URL_PKG_PREFIXES,
						Configuration.getJndiServerPkgPrefix());
				try {
        				Context context = NamingContextPool.getNamingContext(environment);
        				try {
        					repository = (Repository) context.lookup(Configuration.getJcrStoreJNDIPath());
        				} catch (NamingException e) {
        					throw new RepositoryException(e);
        				} finally {
        				    NamingContextPool.releaseNamingContext(context) ;
        				}
                                } catch (NamingContextException nce) {
                                    throw new RepositoryException(nce);
                                }
			}
			// If no JNDI path is specified, init the repository with default
			// settings based on the respository.xml in the conf directory.
			else {
				try {
					ClassLoader loader = Thread.currentThread().getContextClassLoader();
					URL url = loader.getResource("repository.xml");
					URI uri = new URI(url.toString());
					File repositoryConfig = new File(uri);
					ServerConfig config = ServerConfigLocator.locate();
					File dataDir = config.getServerDataDir();
					File repositoryFolder = new File(dataDir, "repository");
					repository = initRepository(repositoryConfig, repositoryFolder);
				} catch (URISyntaxException e) {
					throw new RepositoryException(e);
				}
			}
		}
		return repository;
	}

	/**
	 * Open a session on the repository.
	 * 
	 * @return
	 * @throws RepositoryException
	 */
	public Session newRepositorySession() throws RepositoryException {
		String username = Configuration.getJcrStoreUsername();
		String password = Configuration.getJcrStorePassword();
		if (StringUtils.isEmpty(username) || StringUtils.isEmpty(password)) {
			throw new RepositoryException(
					"JCR username and password must be specified in configuration.");
		}
		return getRepository().login(new SimpleCredentials(username, password.toCharArray()));
	}

	/**
	 * Get the existing repository session or create a new one.
	 * 
	 * @return
	 * @throws RepositoryException
	 */
	public Session getSharedSession() throws RepositoryException {
		if (session == null) {
			session = newRepositorySession();
		}
		return session;
	}
}