/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author daniel.brum@jboss.com
 */


package org.jboss.internal.soa.esb.persistence.format;

import java.util.concurrent.ConcurrentHashMap;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.services.persistence.MessageStore;
import org.jboss.soa.esb.services.persistence.RedeliverStore;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.Configurable;
import org.jboss.soa.esb.ConfigurationException;

/**
 * 
 * @author kurt.stam@jboss.com
 *
 */
public class MessageStoreFactory {
	
	private final ConcurrentHashMap<String, MessageStore> messageStores = new ConcurrentHashMap<String, MessageStore>();
	private static MessageStoreFactory theFactory;
    private static Logger _log = Logger.getLogger(MessageStoreFactory.class);
    private static final ConfigTree DEFAULT_STORE_CONFIG = new ConfigTree("default-config");

    public MessageStoreFactory() {
		reset();
	}
    /**
     * Returns a the default MessageStore implementation (Database).
     * @return MessageStore
     */
    public MessageStore getMessageStore() {
        return getMessageStore(MessageStore.DEFAULT);
    }
    
    /**
     * Returns a the default MessageStore implementation (Database).
     * @return MessageStore
     */
    public RedeliverStore getRedeliverStore() {
        return (RedeliverStore) getMessageStore(MessageStore.DEFAULT);
    }
    
	/**
     * Returns an instance of a MessageStore, of the type 'className'. Multiple instances of the same
     * type of MessageStore will be created for different sets of properties.
     *
     * @param className - classname of the message store.
     * @return MessageStore
	 */
	public MessageStore getMessageStore(String className) {
        return getMessageStore(className, DEFAULT_STORE_CONFIG);
	}
	
	public void reset ()
	{
		messageStores.clear();
    }
    /**
     * Instantiates a 
     * @param className
     * @return
     */
    private MessageStore getMessageStoreInstance(String className) 
    {
        MessageStore messageStore=null;
		try {
			Class messageStoreClass = ClassUtil.forName(className, getClass());
			messageStore = (MessageStore) messageStoreClass.newInstance();
		} catch (ClassNotFoundException ex) {
			_log.error(ex.getMessage(), ex);
		} catch (IllegalAccessException ex) {
            _log.error(ex.getMessage(), ex);
		} catch (InstantiationException ex) {
            _log.error(ex.getMessage(), ex);
		}
        return messageStore;
	}
    
    public static  MessageStoreFactory getInstance()
    {
        if (theFactory==null) {
            theFactory = new MessageStoreFactory();
        }
        return theFactory;
    }

    public MessageStore getMessageStore(String className, ConfigTree config) {
        if (className == null) {
            className = MessageStore.DEFAULT;
        }
        String messageStoreKey = className;
        MessageStore messageStore = messageStores.get(messageStoreKey);
        if (messageStore == null) {
            messageStore = getMessageStoreInstance(className);
            messageStores.put(messageStoreKey, messageStore);
        }

        if(config != null && messageStore instanceof Configurable) {
            try {
                ((Configurable)messageStore).setConfiguration(config);
            } catch (ConfigurationException e) {
                throw new IllegalStateException("Unable to configure message store '" + className + "'.", e);
            }
        }

        return messageStore;
    }
}
