package org.jboss.soa.esb.samples.trailblazer.loanbroker;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */


import java.util.Properties;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.samples.trailblazer.util.TrailblazerProperties;
import org.jboss.soa.esb.samples.trailblazer.web.WebCustomer;
import org.jboss.soa.esb.client.ServiceInvoker;

/*
 * LoanBroker is responsible for getting customer requests for loans onto the JBoss ESB
 * Requests will come from a JSR-181 pojo web service (@org.jboss.soa.esb.samples.trailblazer.web.LoanBrokerWS)
 * The sequence of events from the LoanBroker are:
 * 1 - Prepare a request for the CreditAgency (transform the customer into a XML representation)
 * 2 - Send to CreditAgency and get response (lookup the customer using the SSN (see @CustomerMasterFile)
 * 3 - Prepare a LoanRequest for each of the banks in their unique data structures and send
 * 4 - Collect the response(s) from the bank(s) and send an email to the customer with the quote offers
 */

public class LoanBroker {

    private static Logger logger = Logger.getLogger(LoanBroker.class);
    //used to locate our entries in the trailblazer-properties
    private final String SERVICE_NAME = "request.service.epr.name";
    private final String SERVICE_CAT = "request.service.category";
    private final String JMS_BANK_QUEUE = "jmsbank.queue";
    private Properties properties = new TrailblazerProperties();
    private ServiceInvoker serviceInvoker; 

    public LoanBroker() {
        String serviceCategoryName = properties.getProperty(SERVICE_CAT, "tbCreditAGency");
        String serviceName = properties.getProperty(SERVICE_NAME, "creditagency");

        try {
            serviceInvoker = new ServiceInvoker(serviceCategoryName, serviceName);
        } catch (Exception e) {
            throw new RuntimeException("Failed to create ServiceInvoker for '" + serviceCategoryName + ":" + serviceName + "'.", e);
        }
    }

    public void processLoanRequest(WebCustomer wCustomer) {

        Customer customer = getCustomer(wCustomer);
        //keep the customer in a file someplace for later use, if needed
        CustomerMasterFile.addCustomer(String.valueOf(customer.ssn), customer);

        //step 1 - send to credit agency for credit score if available

        sendToCreditAgency(customer);

        //added a pause here to give the creditagency some time to reply
        try {
            Thread.sleep(5000);
        } catch (InterruptedException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        //step 2 - send to JMS Bank - async
        System.out.println("sending to JMS Bank...");
        sendToJMSBank(customer);

    }

    private Customer getCustomer(WebCustomer wCustomer) {
        Customer customer = new Customer();
        customer.ssn = wCustomer.getSsn();
        customer.name = wCustomer.getName();
        customer.address = wCustomer.getAddress();
        customer.email = wCustomer.getEmail();
        customer.salary = wCustomer.getSalary();
        customer.loanAmount = wCustomer.getLoanAmount();
        customer.loanDuration = wCustomer.getLoanDuration();
        customer.creditScore = 0;

        return customer;
    }

    //uses plain JMS to send a text message
    private void sendToJMSBank(Customer customer) {
        try {
            //create a Quote Request that the bank expects
            QuoteRequest quote = new QuoteRequest();
            quote.ssn = customer.ssn;
            quote.amount = (int) customer.loanAmount;
            quote.creditScore = customer.creditScore;
            quote.creditHistoryLen = 0;    //not sure who added this one
            quote.term = customer.loanDuration;
            quote.customerEmail = customer.email;

            SendJMSMessage sender = new SendJMSMessage();
            String queue = properties.getProperty(JMS_BANK_QUEUE);
            sender.setupConnection(queue);
            logger.debug("setting up jms connection to queue: " + queue);
            logger.debug("sending text message to bank: " + quote.toString());
            sender.sendAMessage(quote.getCSV());
            sender.stop();
        } catch (Exception e) {
            logger.error(e);
        }
    }

    private int sendToCreditAgency(Customer customer) {
        Message message = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
        Message replyMessage;
        int score = 0;

        try {
            logger.debug("sending to credit agency");

            //set the customer inside the message - csv format for now to test
            message.getBody().add(customer.getCSV());
            replyMessage = serviceInvoker.deliverSync(message, 5000);
            if (replyMessage != null) {
                logger.info("received reply from creditAgency action: " + replyMessage.getBody().get());
                String xx = (String) replyMessage.getBody().get();
                score = Integer.parseInt(xx.trim());
            } else {
                logger.debug("reply not received from credit agency - setting a value of 5");
                score = 5;
            }
        } catch (Exception ex2) {
            logger.error("exception occured: " + ex2);
        }

        return score;
    }


}
