/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.services.registry;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.InputStream;
import java.sql.DriverManager;
import java.sql.Statement;
import java.util.Collection;
import java.util.Iterator;
import java.util.Properties;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.testutils.FileUtil;
import org.jboss.soa.esb.testutils.HsqldbUtil;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
/**
 * Testing the registry.
 * 
 * @author kstam
 *
 */
public class RegistryUnitTest
{
    private static String mDbDriver;
	private static String mDbUrl;
	private static String mDbUsername;
	private static String mDbPassword;
	private static final String SERVICE_NAME = "Kurts Travel Agency";
	private static final String CATEGORY = "travel";
	
	private static Logger logger = Logger.getLogger(RegistryUnitTest.class);
	
	/**
	 * Testing the registering of an EPR.
	 */
	@Test
	public void publishEPR() 
	{
		EPR epr = new EPR();
		try {
			Registry registry = RegistryFactory.getRegistry();
			registry.registerEPR(CATEGORY, SERVICE_NAME, "Service for traveling", 
					epr, "Specific Service Binding for traveling");
		} catch (RegistryException re) {
			logger.log(Level.ERROR, re.getLocalizedMessage(), re);
			assertTrue(false);
		}
		try {
			Registry registry = RegistryFactory.getRegistry();
			registry.registerEPR(null, SERVICE_NAME, null, 
					epr, null);
		} catch (RegistryException re) {
			logger.log(Level.ERROR, re.getLocalizedMessage(), re);
			assertTrue(false);
		}
		//make sure we can handle some null arguments
		
	}
	/**
	 * This test will try to find all services in the "travel" category.
	 */
	@Test
	public void findService() 
	{
		try {
			Registry registry = RegistryFactory.getRegistry();
			Collection<String> services = registry.findServices(CATEGORY);
			for (Iterator i=services.iterator();i.hasNext();) {
				String serviceName = (String) i.next();
				logger.log(Level.INFO, "Found service: " + serviceName);
				assertEquals(SERVICE_NAME, serviceName);
			}
		} catch (RegistryException re) {
			logger.log(Level.ERROR, re.getLocalizedMessage(), re);
			assertTrue(false);
		}
	}
	/**
	 * Find the EPRs for this service.
	 */
	@Test
	public void findEPRs() throws ServiceNotFoundException
	{
		try {
			Registry registry = RegistryFactory.getRegistry();
			Collection<EPR> eprs = registry.findEPRs(CATEGORY, SERVICE_NAME);
			for (Iterator i=eprs.iterator();i.hasNext();) {
				EPR epr = (EPR) i.next();
				logger.log(Level.INFO, "Found epr: " + epr);
			}
		} catch (RegistryException re) {
			logger.log(Level.ERROR, re.getLocalizedMessage(), re);
			assertTrue(false);
		}
	}
	@Test
	public void unregisterEPR() 
	{
		try {
			Registry registry = RegistryFactory.getRegistry();
			EPR eprToBeRemoved = new EPR();
			registry.unRegisterEPR(CATEGORY, SERVICE_NAME, eprToBeRemoved);
//			Now make sure this EPR is really gone
			Collection<EPR> eprs = registry.findEPRs(CATEGORY, SERVICE_NAME);
			logger.log(Level.INFO, "Number of Bindings left should be 0, and is: " + eprs.size());
			assertEquals(eprs.size(),0);
		} catch (Exception re) {
			logger.log(Level.ERROR, re.getLocalizedMessage(), re);
			assertTrue(false);
		}
	}
	@Test
	public void unregisterService() 
	{
		try {
			Registry registry = RegistryFactory.getRegistry();
			registry.unRegisterService(CATEGORY, SERVICE_NAME);
			registry.unRegisterService(null, SERVICE_NAME);
			//Make sure it's really gone
			Collection<String> services = registry.findAllServices();
			logger.log(Level.INFO, "Number of Services left should be 0, and is: " + services.size());
			assertEquals(services.size(),0);
		} catch (Exception re) {
			logger.log(Level.ERROR, re.getLocalizedMessage(), re);
			assertTrue(false);
		}
	}
	/**
	 * Setup the database.
	 * @throws Exception
	 */
	@BeforeClass
	public static void runBeforeAllTests() {
		try {
			TestEnvironmentUtil.setESBPropertiesFileToUse();
			//Set the juddi properties file in System so juddi will pick it up later and use the test values.
			String juddiPropertiesFile = "/org/jboss/soa/esb/services/registry/juddi-unittest.properties";
			System.setProperty("juddi.propertiesFile", juddiPropertiesFile);
			//Read this properties file to get the db connection string
			Properties props = new Properties();
			InputStream inStream = Class.class.getResourceAsStream(juddiPropertiesFile);
			props.load(inStream);
			mDbDriver    = props.getProperty("juddi.jdbcDriver");
			mDbUrl       = props.getProperty("juddi.jdbcUrl");
			mDbUsername  = props.getProperty("juddi.jdbcUsername");
			mDbPassword  = props.getProperty("juddi.jdbcPassword");
			
			String database="not tested yet";
			if ("org.hsqldb.jdbcDriver".equals(mDbDriver)) {
				database = "hsqldb";
				//Bring up hsql on default port 9001
				HsqldbUtil.startHsqldb(TestEnvironmentUtil.getUserDir() + "build/hsqltestdb", "juddi");
			} else if ("com.mysql.jdbc.Driver".equals(mDbDriver)) {
				database = "mysql";
			} //add and test your own database..
			
			//Get the registry-schema create scripts
			String sqlDir = TestEnvironmentUtil.getUserDir() + "install/jUDDI-registry/sql/" + database + "/";
			//Drop what is there now, if exists. We want to start fresh.
			String sqlDropCmd      = FileUtil.readTextFile(new File(sqlDir + "drop_database.sql")).replaceAll("\\$\\{prefix}", "");
            String resource = "juddi-sql/" + database + "/create_database.sql";
            InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(resource);
            String sqlCreateCmd    = FileUtil.readStream(is).trim().replaceAll("\\$\\{prefix}", "");
			String sqlInsertPubCmd = FileUtil.readTextFile(new File(sqlDir + "import.sql")).trim().replaceAll("\\$\\{prefix}", "");
			
			try {
				Class.forName(mDbDriver);
			} catch (Exception e) {
				logger.error("ERROR: failed to load " + database + " JDBC driver.", e);
				return;
			}
			java.sql.Connection con = DriverManager.getConnection(mDbUrl, mDbUsername, mDbPassword);
			Statement stmnt = con.createStatement();
			logger.debug("Dropping the schema if exist");
			stmnt.execute(sqlDropCmd);
			logger.debug("Creating the juddi-schema");
			stmnt.execute(sqlCreateCmd);
			logger.debug("Adding the jbossesb publisher");
			stmnt.execute(sqlInsertPubCmd);
		} catch (Exception e) {
			logger.error("We should stop testing, since we don't have a db.", e);
			assertTrue(false);
		}
	}
	/**
	 * Shutdown the database
	 * @throws Exception
	 */
	@AfterClass
	public static void runAfterAllTests() throws Exception {
		if ("org.hsqldb.jdbcDriver".equals(mDbDriver)) {
			HsqldbUtil.stopHsqldb(mDbUrl, mDbUsername, mDbPassword);
		}
	}

	public static junit.framework.Test suite() {
		return new JUnit4TestAdapter(RegistryUnitTest.class);
	}

	}
	
