/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.rosetta.pooling;

import static org.junit.Assert.assertEquals;

import java.util.Properties;

import javax.jms.Session;
import javax.naming.Context;

import junit.framework.JUnit4TestAdapter;

import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPool;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPoolContainer;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.Environment;
import org.junit.Before;
import org.junit.Test;

/**
 * @author kstam
 * @author Daniel Bevenius
 *
 */
public class JmsConnectionPoolingIntegrationTest {
	
	private Properties environment;
	
	@Before
	public void setup()
	{
		environment = getEnvironment();
	}
    
    @Test
    public void testPoolAndConnectionCreation()  throws Exception
    {
        JmsConnectionPool jmsConnectionPool = null;
        
        jmsConnectionPool = JmsConnectionPoolContainer.getPool(environment,"ConnectionFactory", JMSEpr.QUEUE_TYPE);
        assertEquals(0, jmsConnectionPool.getSessionsInPool());
        //Open 3 concurrent sessions
        Session session1 = jmsConnectionPool.getQueueSession();
        assertEquals(1, jmsConnectionPool.getSessionsInPool());
        Session session2 = jmsConnectionPool.getQueueSession();
        assertEquals(2, jmsConnectionPool.getSessionsInPool());
        Session session3 = jmsConnectionPool.getQueueSession();
        assertEquals(3, jmsConnectionPool.getSessionsInPool());
        //Close them
        jmsConnectionPool.closeSession(session1);
        jmsConnectionPool.closeSession(session2);
        jmsConnectionPool.closeSession(session3);
        assertEquals(3, jmsConnectionPool.getSessionsInPool());
        //destroy this pool
        jmsConnectionPool.removeSessionPool();
        assertEquals(0, jmsConnectionPool.getSessionsInPool());
        assertEquals(0, JmsConnectionPoolContainer.getNumberOfPools());
        
        //Use it again and add one session
        jmsConnectionPool.getQueueSession();
        assertEquals(1, jmsConnectionPool.getSessionsInPool());
        assertEquals(1, JmsConnectionPoolContainer.getNumberOfPools());
        //I should be able to remove the entire pool and have it do closing
        //of the session.
        jmsConnectionPool.removeSessionPool();
    }
    
    @Test
    public void testCreateSecondPool() throws Exception
    {
        JmsConnectionPool jmsConnectionPool1 = JmsConnectionPoolContainer.getPool(environment, "ConnectionFactory", JMSEpr.QUEUE_TYPE);
        jmsConnectionPool1 = JmsConnectionPoolContainer.getPool(environment, "ConnectionFactory", JMSEpr.QUEUE_TYPE);
        //This should be the same pool
        assertEquals(1, JmsConnectionPoolContainer.getNumberOfPools());
    
        JmsConnectionPool jmsConnectionPool2 = JmsConnectionPoolContainer.getPool(environment, "ConnectionFactory", JMSEpr.TOPIC_TYPE);
        //This should be a different pool, so now we should have 2.
        assertEquals(2, JmsConnectionPoolContainer.getNumberOfPools());
        
        JmsConnectionPool jmsConnectionPool3 = JmsConnectionPoolContainer.getPool(null, "ConnectionFactory", JMSEpr.TOPIC_TYPE);
        //This should be a different pool, so now we should have 3.
        assertEquals(3, JmsConnectionPoolContainer.getNumberOfPools());
        
        //Now lets cleanup after ourselves
        jmsConnectionPool3.removeSessionPool();
        assertEquals(2, JmsConnectionPoolContainer.getNumberOfPools());
        
        jmsConnectionPool1.removeSessionPool();
        jmsConnectionPool2.removeSessionPool();
        jmsConnectionPool3.removeSessionPool();
        assertEquals(0, JmsConnectionPoolContainer.getNumberOfPools()); 
    }
    
    @Test
    public void testPoolAndSessionsWithAcknowledgeMode()  throws Exception
    {
        JmsConnectionPool jmsConnectionPool = JmsConnectionPoolContainer.getPool(environment,"ConnectionFactory", JMSEpr.QUEUE_TYPE);
        assertEquals(0, jmsConnectionPool.getSessionsInPool());
        
        Session autoAckSession1 = jmsConnectionPool.getQueueSession(Session.AUTO_ACKNOWLEDGE);
        assertEquals(Session.AUTO_ACKNOWLEDGE, autoAckSession1.getAcknowledgeMode());
        assertEquals(1, jmsConnectionPool.getSessionsInPool());
        assertEquals(1, jmsConnectionPool.getSessionsInPool(Session.AUTO_ACKNOWLEDGE));
        
        Session autoAckSession2 = jmsConnectionPool.getQueueSession(Session.AUTO_ACKNOWLEDGE);
        assertEquals(Session.AUTO_ACKNOWLEDGE, autoAckSession2.getAcknowledgeMode());
        assertEquals(2, jmsConnectionPool.getSessionsInPool());
        assertEquals(2, jmsConnectionPool.getSessionsInPool(Session.AUTO_ACKNOWLEDGE));
        
        Session clientAckSession1 = jmsConnectionPool.getQueueSession(Session.CLIENT_ACKNOWLEDGE);
        assertEquals(Session.CLIENT_ACKNOWLEDGE, clientAckSession1.getAcknowledgeMode());
        assertEquals(3, jmsConnectionPool.getSessionsInPool());
        assertEquals(1, jmsConnectionPool.getSessionsInPool(Session.CLIENT_ACKNOWLEDGE));
        
        Session clientAckSession2 = jmsConnectionPool.getQueueSession(Session.CLIENT_ACKNOWLEDGE);
        assertEquals(Session.CLIENT_ACKNOWLEDGE, clientAckSession2.getAcknowledgeMode());
        assertEquals(4, jmsConnectionPool.getSessionsInPool());
        assertEquals(2, jmsConnectionPool.getSessionsInPool(Session.CLIENT_ACKNOWLEDGE));
        
        Session dupsOkAcSession1 = jmsConnectionPool.getQueueSession(Session.DUPS_OK_ACKNOWLEDGE);
        assertEquals(Session.DUPS_OK_ACKNOWLEDGE, dupsOkAcSession1.getAcknowledgeMode());
        assertEquals(5, jmsConnectionPool.getSessionsInPool());
        assertEquals(1, jmsConnectionPool.getSessionsInPool(Session.DUPS_OK_ACKNOWLEDGE));
        
        Session dupsOkAcSession2 = jmsConnectionPool.getQueueSession(Session.DUPS_OK_ACKNOWLEDGE);
        assertEquals(Session.DUPS_OK_ACKNOWLEDGE, dupsOkAcSession2.getAcknowledgeMode());
        assertEquals(6, jmsConnectionPool.getSessionsInPool());
        assertEquals(2, jmsConnectionPool.getSessionsInPool(Session.DUPS_OK_ACKNOWLEDGE));
        
        //Close them
        jmsConnectionPool.closeSession(autoAckSession1);
        assertEquals(6, jmsConnectionPool.getSessionsInPool());
        assertEquals(1, jmsConnectionPool.getFreeSessionsInPool(Session.AUTO_ACKNOWLEDGE));
        assertEquals(1, jmsConnectionPool.getInUseSessionsInPool(Session.AUTO_ACKNOWLEDGE));
        
        jmsConnectionPool.closeSession(autoAckSession2);
        assertEquals(6, jmsConnectionPool.getSessionsInPool());
        assertEquals(2, jmsConnectionPool.getFreeSessionsInPool(Session.AUTO_ACKNOWLEDGE));
        assertEquals(0, jmsConnectionPool.getInUseSessionsInPool(Session.AUTO_ACKNOWLEDGE));
        
        jmsConnectionPool.closeSession(clientAckSession1);
        assertEquals(6, jmsConnectionPool.getSessionsInPool());
        assertEquals(1, jmsConnectionPool.getFreeSessionsInPool(Session.CLIENT_ACKNOWLEDGE));
        assertEquals(1, jmsConnectionPool.getInUseSessionsInPool(Session.CLIENT_ACKNOWLEDGE));
        
        jmsConnectionPool.closeSession(clientAckSession2);
        assertEquals(6, jmsConnectionPool.getSessionsInPool());
        assertEquals(2, jmsConnectionPool.getFreeSessionsInPool(Session.CLIENT_ACKNOWLEDGE));
        assertEquals(0, jmsConnectionPool.getInUseSessionsInPool(Session.CLIENT_ACKNOWLEDGE));
        
        jmsConnectionPool.closeSession(dupsOkAcSession1);
        assertEquals(6, jmsConnectionPool.getSessionsInPool());
        assertEquals(1, jmsConnectionPool.getFreeSessionsInPool(Session.DUPS_OK_ACKNOWLEDGE));
        assertEquals(1, jmsConnectionPool.getInUseSessionsInPool(Session.DUPS_OK_ACKNOWLEDGE));
        
        jmsConnectionPool.closeSession(dupsOkAcSession2);
        assertEquals(6, jmsConnectionPool.getSessionsInPool());
        assertEquals(2, jmsConnectionPool.getFreeSessionsInPool(Session.DUPS_OK_ACKNOWLEDGE));
        assertEquals(0, jmsConnectionPool.getInUseSessionsInPool(Session.DUPS_OK_ACKNOWLEDGE));
        
        jmsConnectionPool.removeSessionPool();
        assertEquals(0, jmsConnectionPool.getSessionsInPool());
        assertEquals(0, JmsConnectionPoolContainer.getNumberOfPools());
        
        jmsConnectionPool.getQueueSession();
        assertEquals(1, jmsConnectionPool.getSessionsInPool());
        assertEquals(1, JmsConnectionPoolContainer.getNumberOfPools());
        
        jmsConnectionPool.removeSessionPool();
    }
    
    public Properties getEnvironment()
    {
    	  Properties environment = new Properties();
          environment.setProperty(Context.PROVIDER_URL, Environment.JBOSS_PROVIDER_URL);
          environment.setProperty(Context.INITIAL_CONTEXT_FACTORY, Environment.JBOSS_INITIAL_CONTEXT_FACTORY);
          environment.setProperty(Context.URL_PKG_PREFIXES, Environment.JBOSS_URL_PKG_PREFIX);
          return environment;
    }
    
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(JmsConnectionPoolingIntegrationTest.class);
    }
}
