/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config;

import java.io.File;
import java.io.IOException;
import java.util.List;

import junit.framework.TestCase;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.listeners.config.Generator.XMLBeansModel;
import org.jboss.soa.esb.listeners.config.xbeanmodel.JmsBusDocument.JmsBus;
import org.jboss.soa.esb.listeners.config.xbeanmodel.JmsListenerDocument.JmsListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel.JmsMessageFilterDocument.JmsMessageFilter;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ServiceDocument.Service;
import org.jboss.soa.esb.listeners.config.xbeanmodel.JmsProviderType;
import org.jboss.soa.esb.listeners.config.xbeanmodel.Listener;
import org.jboss.soa.esb.testutils.FileUtil;

/**
 * Generator unit tests.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class GeneratorUnitTest extends TestCase {
	
	private Logger log = Logger.getLogger( GeneratorUnitTest.class );
	
	private File outdir = new File("test-outdir");
	private File gatewayConfig;
	private File awareConfig;

	public void test_constructor() throws ConfigurationException, IOException {
		try {
			new Generator(null);
			fail("Expected IllegalArgumentException.");
		} catch (IllegalArgumentException e) {
			// OK
		}
	}
	
	public void test_generate_badagrs() throws ConfigurationException, IOException {
		test_generate_badargs(null, "null 'outdir' arg in method call");
		test_generate_badargs(new File("build.xml"), "'outdir' exists, but is not a directory");
	}
	
	public void test_generate_files() throws ConfigurationException, IOException {
		test_generate_files("jbossesb_config_01.xml");
		test_generate_files("jbossesb_config_02.xml");
	}
		
	public void test_generate_files(String filename) throws ConfigurationException, IOException {
		Generator generator = new Generator(getClass().getResourceAsStream(filename));
		String expectedGatewayConfig = new String(StreamUtils.readStream(getClass().getResourceAsStream("jbossesb_config_01_gateways.xml"))).trim();
		String expectedESBAwareConfig = new String(StreamUtils.readStream(getClass().getResourceAsStream("jbossesb_config_01_esbaware.xml"))).trim();
		String actualGatewayConfig;
		String actualESBAwareConfig;
		
		expectedGatewayConfig = removeEOL(expectedGatewayConfig);
		expectedESBAwareConfig = removeEOL(expectedESBAwareConfig);
		
		generator.generate(outdir);
		
		gatewayConfig = new File(outdir, Generator.ESB_CONFIG_GATEWAY_XML_FILE);
		assertTrue(gatewayConfig.exists());
		actualGatewayConfig = FileUtil.readTextFile(gatewayConfig);
		log.debug("actual  [" + removeEOL(actualGatewayConfig) + "]");
        log.debug("expected[" + expectedGatewayConfig + "]");
		assertEquals(expectedGatewayConfig, removeEOL(actualGatewayConfig));
		
		awareConfig = new File(outdir, Generator.ESB_CONFIG_XML_FILE);
		assertTrue(awareConfig.exists());
		actualESBAwareConfig = FileUtil.readTextFile(awareConfig);
		log.debug("[" + actualESBAwareConfig + "]");
		actualESBAwareConfig = removeEOL(actualESBAwareConfig);
		boolean isIndentical = expectedESBAwareConfig.equals(actualESBAwareConfig);
		log.debug("Strings identical=" + isIndentical);
		assertEquals(expectedESBAwareConfig, actualESBAwareConfig);
	}

	public void test_bad_base_listener_busref() throws ConfigurationException, IOException {
		Generator generator = new Generator(getClass().getResourceAsStream("jbossesb_config_03.xml"));
		
		try {
			generator.generate(outdir);
			fail("Expected ConfigurationException");
		} catch(ConfigurationException e) {
			if(!e.getMessage().equals("The base Listener config [JMS-ESBListener] must reference a base Bus config type (<bus>).")) {
				fail("Wrong exception");
			}
		}
	}
	
	public static String removeEOL(String string) {
		StringBuffer stringBuf = new StringBuffer(string.length());

		for(int i = 0; i < string.length(); i++) {
			char character = string.charAt(i);
			
			if(character != '\r' && character != '\n') {
				stringBuf.append(character);
			}
		}
		
		return stringBuf.toString();
	}

	public void test_model() throws ConfigurationException, IOException {
		Generator generator = new Generator(getClass().getResourceAsStream("jbossesb_config_01.xml"));
		XMLBeansModel model = generator.getModel();

		JmsBus bus = (JmsBus)model.getBus("server1-jms");
		JmsProviderType provider = (JmsProviderType) model.getProvider(bus);
		assertEquals("server1-jms", bus.getBusid());
		assertEquals("com.xyz.provider.XYZConnectionFactory", provider.getConnectionFactory());
		assertEquals("com.xyz.provider.NamingContextFactory", provider.getJndiContextFactory());
		assertEquals("xyz://server1:9876", provider.getJndiURL());
		assertEquals("com.xyz", provider.getJndiPkgPrefix());
		JmsMessageFilter busDestination = bus.getJmsMessageFilter();
		assertEquals("queue/A", busDestination.getDestName());
		assertEquals(JmsMessageFilter.DestType.QUEUE, busDestination.getDestType());
		assertEquals("service='Reconciliation'", busDestination.getSelector());

		List<Listener> gateways = model.getGatewayListeners();
		assertEquals(2, gateways.size());
		JmsListener gatewayListener = (JmsListener)gateways.get(0);
		assertEquals("Bank-JMS-Gateway", gatewayListener.getName());
		assertEquals("server1-jms", gatewayListener.getBusidref());
		assertTrue(gatewayListener.getIsGateway());
		assertEquals(1, gatewayListener.getMaxThreads());

		List<Listener> awareListeners = model.getESBAwareListeners();
		assertEquals(4, awareListeners.size());
		JmsListener awareListener = (JmsListener)awareListeners.get(0);
		assertEquals("Bank-Listener", awareListener.getName());
		assertEquals("local-jms", awareListener.getBusidref());
		assertTrue(!awareListener.getIsGateway());
		assertEquals(2, awareListener.getMaxThreads());
		JmsMessageFilter listenerDestination = awareListener.getJmsMessageFilter();
		assertEquals("queue/B", listenerDestination.getDestName());
		assertEquals(JmsMessageFilter.DestType.TOPIC, listenerDestination.getDestType());
		assertEquals("service='Reconciliation'", listenerDestination.getSelector());
		
		Listener untypedAwareListener = awareListeners.get(1);
		assertEquals("Bank-Listener-Generic", untypedAwareListener.getName());
		assertEquals("my-generic-bus", untypedAwareListener.getBusidref());
		assertTrue(!untypedAwareListener.getIsGateway());
		
		Service gatewayService = model.getService(gatewayListener);
		Service awareService = model.getService(awareListener);
		assertSame(gatewayService, awareService);
		assertEquals("Bank", awareService.getCategory());
		assertEquals("Reconciliation", awareService.getName());
		assertEquals("Bank Reconciliation Service", awareService.getDescription());
	}

	private void test_generate_badargs(File outdir, String expectedMessage) throws ConfigurationException, IOException {
		Generator generator = new Generator(getClass().getResourceAsStream("jbossesb_config_01.xml"));

		try {
			generator.generate(outdir);
			fail("Expected IllegalArgumentException.");
		} catch(IllegalArgumentException e) {
			// Expected
			if(!e.getMessage().startsWith(expectedMessage)) {
				fail("Unexpected exception message.  Expected message to start with [" + expectedMessage + "].  Message was [" + e.getMessage() + "].");
			}
		}
	}

	/* (non-Javadoc)
	 * @see junit.framework.TestCase#tearDown()
	 */
	@Override
	protected void tearDown() throws Exception {
		if(awareConfig != null && awareConfig.exists()) {
			awareConfig.delete();
		}
		if(gatewayConfig != null && gatewayConfig.exists()) {
			gatewayConfig.delete();
		}
		if(outdir.exists()) {
			outdir.delete();
		}
	}
}
