/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners;

import java.io.IOException;
import java.io.InputStream;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.config.ESBAwareGenerator;
import org.jboss.soa.esb.listeners.config.GatewayGenerator;
import org.jboss.soa.esb.listeners.config.Generator;
import org.w3c.dom.Document;

/**
 * Test utility class for generating a listener configuration from an xsd based config.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class ListenerConfigUtil {

    /**
     * Generate an ESB aware listener config from the supplied XSD based configuration.
     *
     * @param config       The XSD based config stream.
     * @param listenerName The name of listener configuration.  This is the name supplied in
     *                     the "name" attribute on the listener config.
     * @return The listener ConfigTree config.
     * @throws ConfigurationException Bad listener ESB configuration.
     * @throws IOException            Unable to read the ESB listener configuration.
     */
    public static ConfigTree getESBAwareListenerConfig(InputStream config, String listenerName) throws ConfigurationException, IOException {
        Generator.XMLBeansModel model = Generator.parseConfig(config);

        ESBAwareGenerator awareGenerator = new ESBAwareGenerator(model);
        Document awareConfig = awareGenerator.generate();

        return getConfigTree(awareConfig, listenerName);
    }

    /**
     * Generate a Gateway listener config from the supplied XSD based configuration.
     *
     * @param config       The XSD based config stream.
     * @param listenerName The name of listener configuration.  This is the name supplied in
     *                     the "name" attribute on the listener config.
     * @return The listener ConfigTree config.
     * @throws ConfigurationException Bad listener ESB configuration.
     * @throws IOException            Unable to read the ESB listener configuration.
     */
    public static ConfigTree getGatewayListenerConfig(InputStream config, String listenerName) throws ConfigurationException, IOException {
        Generator.XMLBeansModel model = Generator.parseConfig(config);

        GatewayGenerator gatewayGenerator = new GatewayGenerator(model);
        Document gatewayConfig = gatewayGenerator.generate();

        return getConfigTree(gatewayConfig, listenerName);
    }

    private static ConfigTree getConfigTree(Document configDoc, String listenerName) {
        ConfigTree config = ConfigTree.fromElement(configDoc.getDocumentElement());

        return config.getFirstChild(listenerName);
    }
}
