/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.command;

import java.util.ArrayList;
import java.util.List;

import junit.framework.TestCase;

import org.jboss.internal.soa.esb.command.CommandQueue;
import org.jboss.internal.soa.esb.command.CommandQueueException;
import org.jboss.internal.soa.esb.command.InMemoryCommandQueue;
import org.jboss.soa.esb.helpers.ConfigTree;

public class InMemoryCommandQueueUnitTest extends TestCase {

	public void test_args() throws CommandQueueException {
		InMemoryCommandQueue commandQueue = new InMemoryCommandQueue();
		
		try {
			commandQueue.open(null);
			fail("Expected IllegalArgumentException.");
		} catch (IllegalArgumentException e) {
			// OK
		}

		ConfigTree config = new ConfigTree("config");
		try {
			commandQueue.open(config);
			fail("Expected CommandQueueException.");
		} catch (CommandQueueException e) {
			// OK
		}
	}
	
	public void test_queue_open_close() throws CommandQueueException {
		ConfigTree config = new ConfigTree("config");
		InMemoryCommandQueue commandQueue = new InMemoryCommandQueue();

		config.setAttribute(InMemoryCommandQueue.COMMAND_QUEUE_NAME, "test-queue");
		assertEquals(null, InMemoryCommandQueue.getQueue("test-queue"));
		commandQueue.open(config);
		assertEquals(commandQueue, InMemoryCommandQueue.getQueue("test-queue"));
		commandQueue.close();
		assertEquals(null, InMemoryCommandQueue.getQueue("test-queue"));
	}
	
	public void test_queue_receive() throws CommandQueueException, InterruptedException {
		ConfigTree config = new ConfigTree("config");
		InMemoryCommandQueue commandQueue = new InMemoryCommandQueue();

		// receive should fail if the queue hasn't been opened yet...
		try {
			commandQueue.receiveCommand(0);
			fail("Expected CommandQueueException.");
		} catch (CommandQueueException e) {
			// OK
		}
		
		config.setAttribute(InMemoryCommandQueue.COMMAND_QUEUE_NAME, "test-queue");
		commandQueue.open(config);
		
		// Start the consumer thread - it will receive the commands from the queue.
		CommandConsumerThread consumerThread = new CommandConsumerThread(commandQueue);
		consumerThread.start();
		
		// Make sure the thread is running.
		assertTrue(consumerThread.isRunning);
		
		commandQueue.addCommand("command1");
		assertCommandReceived(consumerThread, "command1", 0);
		commandQueue.addCommand("command2");
		assertCommandReceived(consumerThread, "command2", 1);
		commandQueue.addCommand("command3");
		assertCommandReceived(consumerThread, "command3", 2);
		
		// Stop the queue thread...
		commandQueue.addCommand("stop");
		Thread.sleep(50);
		assertTrue(!consumerThread.isRunning);  // this flag being reset proves the stop command was consumed and so the queue is really working
		assertEquals(4, consumerThread.unblockCount); // Should have unblocked 4 times - once for each command.
		
		// receive should fail if the queue has been closed...
		commandQueue.close();
		try {
			commandQueue.receiveCommand(0);
			fail("Expected CommandQueueException.");
		} catch (CommandQueueException e) {
			// OK
		}
	}
	
	private void assertCommandReceived(CommandConsumerThread consumerThread, String expected, int index) throws InterruptedException {
		Thread.sleep(50);
		assertEquals("Received commands queue is not the expected length.", index + 1, consumerThread.commandsReceived.size());
		assertEquals("Command not found as last added command.", expected, consumerThread.commandsReceived.get(index));
	}

	private class CommandConsumerThread extends Thread {

		private List<String> commandsReceived = new ArrayList<String>();
		private CommandQueue commandQueue;
		private boolean isRunning = true;
		private int unblockCount = 0;
		
		private CommandConsumerThread(CommandQueue commandQueue) {
			this.commandQueue = commandQueue;
		}
		
		@Override
		public void run() {
			String command = null;
			
			while(!"stop".equals(command)) {
				try {
					command = commandQueue.receiveCommand(0);
					commandsReceived.add(command);
				} catch (CommandQueueException e) {
					fail("CommandQueue Exception: " + e.getMessage());
				}
				unblockCount++;
			}
			isRunning = false;
		}		
	}
}
