/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.addressing.eprs;

import static org.jboss.soa.esb.addressing.eprs.JMSEpr.*;
import static org.junit.Assert.assertEquals;
import java.net.URISyntaxException;
import java.util.Properties;

import javax.jms.Session;
import javax.naming.Context;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.addressing.eprs.JMSEpr.AcknowledgeMode;
import org.jboss.soa.esb.couriers.CourierException;
import org.junit.Test;

/**
 * Unit Test class for  JMSEpr
 * 
 * @author <a href="mailto:daniel.bevenius@gmail.com">Daniel Bevenius</a>				
 */
public class JMSEprUnitTest
{
	private String expectedDestination = "/queue/test";
	private String expectedConnectionFactory = "ConnectionFactory";
	private String expectedDestinationType = JMSEpr.QUEUE_TYPE;
	private String expectedSelector = "JMSCorrelationID=1000";
	private static final boolean NON_PERSISTENT = false;
	private static final boolean PERSISTENT = true;
	
	private final Properties nullEnvironment = null;
	
	@Test
	public void contstructor_queue() throws CourierException, URISyntaxException
	{
		JMSEpr jmsEpr = new JMSEpr( expectedDestinationType, expectedDestination , expectedConnectionFactory );
		assertDefaults( jmsEpr.getDestinationName(), jmsEpr.getConnectionFactory(), jmsEpr.getDestinationType() );
	}
	
	@Test
	public void contstructor_props_and_selector() throws CourierException, URISyntaxException
	{
		JMSEpr jmsEpr = new JMSEpr( expectedDestinationType, expectedDestination , expectedConnectionFactory,
				nullEnvironment, expectedSelector);
		
		assertDefaults( jmsEpr.getDestinationName(), jmsEpr.getConnectionFactory(), jmsEpr.getDestinationType() );
		assertEquals( expectedSelector, jmsEpr.getMessageSelector() );
	}
	
	@Test
	public void contstructor_persistent() throws CourierException, URISyntaxException
	{
		JMSEpr jmsEpr = new JMSEpr( ONE_ONE_PROTOCOL, expectedDestinationType, expectedDestination , 
				expectedConnectionFactory,
				nullEnvironment, expectedSelector, PERSISTENT);
		
		assertDefaults( jmsEpr.getDestinationName(), jmsEpr.getConnectionFactory(), jmsEpr.getDestinationType() );
		assertEquals( ONE_ONE_PROTOCOL, jmsEpr.getVersion() );
		assertEquals( expectedSelector, jmsEpr.getMessageSelector() );
		assertEquals( PERSISTENT, jmsEpr.getPersistent() );
	}
	
	@Test
	public void contstructor_non_persistent() throws CourierException, URISyntaxException
	{
		JMSEpr jmsEpr = new JMSEpr( ONE_ONE_PROTOCOL, expectedDestinationType, expectedDestination , 
				expectedConnectionFactory,
				nullEnvironment, expectedSelector, NON_PERSISTENT);
		
		assertDefaults( jmsEpr.getDestinationName(), jmsEpr.getConnectionFactory(), jmsEpr.getDestinationType() );
		assertEquals( ONE_ONE_PROTOCOL, jmsEpr.getVersion() );
		assertEquals( expectedSelector, jmsEpr.getMessageSelector() );
		assertEquals( NON_PERSISTENT, jmsEpr.getPersistent() );
	}
	
	@Test
	public void contstructor_acknowledeMode_default() throws CourierException, URISyntaxException
	{
		JMSEpr jmsEpr = new JMSEpr( ONE_ONE_PROTOCOL, expectedDestinationType, expectedDestination , 
				expectedConnectionFactory,
				nullEnvironment, expectedSelector, NON_PERSISTENT);
		
		assertDefaults( jmsEpr.getDestinationName(), jmsEpr.getConnectionFactory(), jmsEpr.getDestinationType() );
		assertEquals( "Default acknowledemode should be AUTO_ACKNOWLEDGE", Session.AUTO_ACKNOWLEDGE, jmsEpr.getAcknowledgeMode() );
	}
	
	@Test
	public void contstructor_acknowledeMode_negative_ackmode_null() throws CourierException, URISyntaxException
	{
		final String acknowledgeModeStr = null;
		JMSEpr jmsEpr = new JMSEpr( ONE_ONE_PROTOCOL, expectedDestinationType, expectedDestination , 
				expectedConnectionFactory,
				nullEnvironment, expectedSelector, NON_PERSISTENT, acknowledgeModeStr);
		
		assertEquals( Session.AUTO_ACKNOWLEDGE, jmsEpr.getAcknowledgeMode() );
	}
	
	@Test
	public void contstructor_acknowledeMode_negative() throws CourierException, URISyntaxException
	{
		JMSEpr jmsEpr = new JMSEpr( ONE_ONE_PROTOCOL, expectedDestinationType, expectedDestination , 
				expectedConnectionFactory,
				nullEnvironment, expectedSelector, NON_PERSISTENT, "BogusAckMode");
		assertEquals( Session.AUTO_ACKNOWLEDGE, jmsEpr.getAcknowledgeMode() );
	}
	
	@Test
	public void contstructor_acknowledeMode_client_ack() throws CourierException, URISyntaxException
	{
		JMSEpr jmsEpr = new JMSEpr( ONE_ONE_PROTOCOL, expectedDestinationType, expectedDestination , 
				expectedConnectionFactory,
				nullEnvironment, expectedSelector, NON_PERSISTENT, AcknowledgeMode.CLIENT_ACKNOWLEDGE.toString());
		
		assertDefaults( jmsEpr.getDestinationName(), jmsEpr.getConnectionFactory(), jmsEpr.getDestinationType() );
		assertEquals( Session.CLIENT_ACKNOWLEDGE, jmsEpr.getAcknowledgeMode() );
	}
	
	@Test
	public void contstructor_with_naming_security_properties() throws CourierException, URISyntaxException
	{
		final String principal = "name";
		final String credential = "password";
		
		final Properties env = new Properties();
		env.put( Context.SECURITY_PRINCIPAL, principal );
		env.put( Context.SECURITY_CREDENTIALS, credential );
		
		JMSEpr jmsEpr = new JMSEpr( ONE_ONE_PROTOCOL, expectedDestinationType, expectedDestination , 
				expectedConnectionFactory,
				env, expectedSelector, NON_PERSISTENT, AcknowledgeMode.CLIENT_ACKNOWLEDGE.toString());
		
		assertEquals( principal, jmsEpr.getJndiEnvironment().getProperty( Context.SECURITY_PRINCIPAL ));
		assertEquals( credential, jmsEpr.getJndiEnvironment().getProperty( Context.SECURITY_CREDENTIALS ));
	}
	
	@Test
	public void contstructor_with_jms_destination_security_negative() throws CourierException, URISyntaxException
	{
		final String username = null;
		final String password = null;
		
		JMSEpr jmsEpr = new JMSEpr( ONE_ONE_PROTOCOL, expectedDestinationType, expectedDestination , 
				expectedConnectionFactory,
				nullEnvironment, expectedSelector, NON_PERSISTENT, AcknowledgeMode.CLIENT_ACKNOWLEDGE.toString(),
				username,
				password);
		
		assertEquals( null, jmsEpr.getJMSSecurityPrincipal() );
		assertEquals( null, jmsEpr.getJMSSecurityCredential() );
	}
	
	@Test
	public void contstructor_with_jms_destination_security() throws CourierException, URISyntaxException
	{
		final String username = "daniel";
		final String password = "password";
		
		JMSEpr jmsEpr = new JMSEpr( ONE_ONE_PROTOCOL, expectedDestinationType, expectedDestination , 
				expectedConnectionFactory,
				nullEnvironment, expectedSelector, NON_PERSISTENT, AcknowledgeMode.CLIENT_ACKNOWLEDGE.toString(),
				username,
				password);
		
		assertEquals( username, jmsEpr.getJMSSecurityPrincipal() );
		assertEquals( password, jmsEpr.getJMSSecurityCredential() );
	}
	
	@Test
	public void acknowledgeEnumTest()
	{
		AcknowledgeMode ackMode = AcknowledgeMode.getAckMode( null );
		assertEquals( AcknowledgeMode.AUTO_ACKNOWLEDGE, ackMode );
		assertEquals( Session.AUTO_ACKNOWLEDGE, ackMode.getAcknowledgeModeInt() );
		
		ackMode = AcknowledgeMode.getAckMode( "bajja" );
		assertEquals( Session.AUTO_ACKNOWLEDGE, ackMode.getAcknowledgeModeInt() );
		
		ackMode = AcknowledgeMode.getAckMode( "-100" );
		assertEquals( Session.AUTO_ACKNOWLEDGE, ackMode.getAcknowledgeModeInt() );
		
		ackMode = AcknowledgeMode.getAckMode( "CLIENT_ACKNOWLEDGE" );
		assertEquals( AcknowledgeMode.CLIENT_ACKNOWLEDGE, ackMode );
		assertEquals( Session.CLIENT_ACKNOWLEDGE, ackMode.getAcknowledgeModeInt() );
		
		ackMode = AcknowledgeMode.getAckMode( "DUPS_OK_ACKNOWLEDGE" );
		assertEquals( AcknowledgeMode.DUPS_OK_ACKNOWLEDGE, ackMode );
		assertEquals( Session.DUPS_OK_ACKNOWLEDGE, ackMode.getAcknowledgeModeInt() );
	}
	
	@Test
	public void contstructor_non_transacted() throws CourierException, URISyntaxException
	{
		JMSEpr jmsEpr = new JMSEpr( ONE_ONE_PROTOCOL, expectedDestinationType, expectedDestination , 
				expectedConnectionFactory,
				nullEnvironment, expectedSelector, 
				NON_PERSISTENT);
		
		assertEquals( false, jmsEpr.getTransacted() );
	}
	
	@Test
	public void contstructor_transacted() throws CourierException, URISyntaxException
	{
		final boolean transacted = true;
		JMSEpr jmsEpr = new JMSEpr( ONE_ONE_PROTOCOL, expectedDestinationType, expectedDestination , 
				expectedConnectionFactory,
				nullEnvironment, expectedSelector, 
				NON_PERSISTENT, transacted);
		
		assertEquals( transacted, jmsEpr.getTransacted() );
	}
	
	private void assertDefaults(final String destination, final String connectionFactory, final String destinationType)
	{
		assertEquals( expectedDestination, destination );
		assertEquals( expectedConnectionFactory, connectionFactory );
		assertEquals( expectedDestinationType, destinationType );
	}
	
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( JMSEprUnitTest.class );
	}

}
