/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.internal.soa.esb.services.registry;

import java.util.ArrayList;
import java.util.List;

import junit.framework.TestCase;

import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.services.registry.Registry;
import org.jboss.soa.esb.services.registry.RegistryException;

import com.arjuna.common.util.propertyservice.PropertyManager;

/**
 * Mock Registry implementation.
 * <p/>
 * Avoids the need to setup HSQLDB etc.
 * <p/>
 * Just call {@link #install()} and {@link #uninstall()} from inside your test setUp and tearDown
 * methods respectfully.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class MockRegistry implements Registry {

    private static final String REGISTRY_IMPEMENTATION_CLASS_ORIGINAL = Environment.REGISTRY_IMPEMENTATION_CLASS + "#Original";
    public static PropertyManager regPropManager = ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE);
    public static List<RepositoryEntry> repository = new ArrayList<RepositoryEntry>();

    /**
     * Install this Mock Registry impl as the registry implementation to be used.
     * <p/>
     * Call this method in the test setUp.
     */
    public static void install() {
        if (regPropManager == null) {
            TestCase.fail("Failed to locate PropertyManager for [" + ModulePropertyManager.REGISTRY_MODULE + "].");
        }
        String currentRegImpl = regPropManager.getProperty(Environment.REGISTRY_IMPEMENTATION_CLASS);

        if (currentRegImpl != null) {
            // Save the current/original.
            regPropManager.setProperty(REGISTRY_IMPEMENTATION_CLASS_ORIGINAL, currentRegImpl);
        }
        regPropManager.setProperty(Environment.REGISTRY_IMPEMENTATION_CLASS, MockRegistry.class.getName());
    }

    /**
     * Uninstall this Mock Registry impl as the registry implementation to be used. Reinstate the reg impl that was
     * specified prior to the install.
     * <p/>
     * Call this method in the test tearDown.
     */
    public static void uninstall() {
        if (regPropManager == null) {
            TestCase.fail("Failed to locate PropertyManager for [" + ModulePropertyManager.REGISTRY_MODULE + "].");
        }
        String originalRegImpl = regPropManager.getProperty(REGISTRY_IMPEMENTATION_CLASS_ORIGINAL);

        if (originalRegImpl != null) {
            // Reset the original.
            regPropManager.setProperty(Environment.REGISTRY_IMPEMENTATION_CLASS, originalRegImpl);
            regPropManager.removeProperty(REGISTRY_IMPEMENTATION_CLASS_ORIGINAL);
        } else {
            // It wasn't set in the first place, so just unset the impl
            regPropManager.removeProperty(Environment.REGISTRY_IMPEMENTATION_CLASS);
        }
        repository = new ArrayList<RepositoryEntry>();
    }

    /**
     * Utility method for registering a courier for a service.
     * <p/>
     * The utility creates an empty EPR against
     *
     * @param category Service category.
     * @param service Service name.
     * @param courier EPR Courier.
     */
    public static void register(String category, String service, MockCourier courier) {
        EPR epr = new EPR();
        register(category, service, epr, courier);
    }

    /**
     * Utility method for registering a service EPR, as well as its courier.
     * @param category Service category.
     * @param service Service name.
     * @param epr Service EPR.
     * @param courier EPR Courier.
     */
    public static void register(String category, String service, EPR epr, MockCourier courier) {
        repository.add(new RepositoryEntry(category, service, "", epr, ""));
        MockCourierFactory.couriers.put(epr, courier);
    }

    /* (non-Javadoc)
      * @see org.jboss.soa.esb.services.registry.Registry#registerEPR(java.lang.String, java.lang.String, java.lang.String, org.jboss.soa.esb.addressing.EPR, java.lang.String)
      */
    public void registerEPR(String serviceCategoryName, String serviceName,
                            String serviceDescription, EPR epr, String eprDescription)
            throws RegistryException {

        repository.add(new RepositoryEntry(serviceCategoryName, serviceName, serviceDescription, epr, eprDescription));
    }

    /* (non-Javadoc)
      * @see org.jboss.soa.esb.services.registry.Registry#unRegisterService(java.lang.String, java.lang.String)
      */
    public void unRegisterService(String category, String serviceName) throws RegistryException {
        unRegisterEPR(category, serviceName, null);
    }

    /* (non-Javadoc)
      * @see org.jboss.soa.esb.services.registry.Registry#unRegisterEPR(java.lang.String, java.lang.String, org.jboss.soa.esb.addressing.EPR)
      */
    public void unRegisterEPR(String serviceCategoryName, String serviceName, EPR epr) throws RegistryException {
        int indexOf = repository.indexOf(new RepositoryEntry(serviceCategoryName, serviceName, null, epr, null));

        if (indexOf == -1) {
            throw new RegistryException("Registry entry [" + serviceCategoryName + "][" + serviceName + "] not found.");
        }
        repository.remove(indexOf);
    }

    /* (non-Javadoc)
      * @see org.jboss.soa.esb.services.registry.Registry#findAllServices()
      */
    public List<String> findAllServices() throws RegistryException {
        List<String> services = new ArrayList<String>();
        for (RepositoryEntry entry : repository) {
            services.add(entry.serviceName);
        }
        return services;
    }

    /* (non-Javadoc)
      * @see org.jboss.soa.esb.services.registry.Registry#findServices(java.lang.String)
      */
    public List<String> findServices(String serviceCategoryName)
            throws RegistryException {
        List<String> services = new ArrayList<String>();
        for (RepositoryEntry entry : repository) {
            if (serviceCategoryName.equals(entry.serviceCategoryName)) {
                services.add(entry.serviceName);
            }
        }
        return services;
    }

    /* (non-Javadoc)
      * @see org.jboss.soa.esb.services.registry.Registry#findEPRs(java.lang.String, java.lang.String)
      */
    public List<EPR> findEPRs(String serviceCategoryName,
                                    String serviceName) throws RegistryException {
        List<EPR> services = new ArrayList<EPR>();
        for (RepositoryEntry entry : repository) {
            if (serviceCategoryName.equals(entry.serviceCategoryName) && serviceName.equals(entry.serviceName)) {
                services.add(entry.epr);
            }
        }
        return services;
    }

    /* (non-Javadoc)
      * @see org.jboss.soa.esb.services.registry.Registry#findEPR(java.lang.String, java.lang.String)
      */
    public EPR findEPR(String serviceCategoryName, String serviceName)
            throws RegistryException {
        for (RepositoryEntry entry : repository) {
            if (serviceCategoryName.equals(entry.serviceCategoryName) && serviceName.equals(entry.serviceName)) {
                return entry.epr;
            }
        }
        return null;
    }

    public static class RepositoryEntry {
        public String serviceCategoryName;
        public String serviceName;
        public String serviceDescription;
        public EPR epr;
        public String eprDescription;

        public RepositoryEntry(String serviceCategoryName, String serviceName, String serviceDescription, EPR epr, String eprDescription) {
            this.serviceCategoryName = serviceCategoryName;
            this.serviceName = serviceName;
            this.serviceDescription = serviceDescription;
            this.epr = epr;
            this.eprDescription = eprDescription;
        }

        public boolean equals(Object obj) {
            if (obj instanceof RepositoryEntry) {
                RepositoryEntry entry = (RepositoryEntry) obj;

                if (serviceCategoryName != null && !serviceCategoryName.equalsIgnoreCase(entry.serviceCategoryName)) {
                    return false;
                }
                if (serviceName != null && !serviceName.equalsIgnoreCase(entry.serviceName)) {
                    return false;
                }
                if (epr != null && epr != entry.epr) {
                    return false;
                }

                return true;
            }

            return false;
        }
    }
}
