/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.util;

import java.io.*;

import org.apache.log4j.Logger;

/**
 * Common file utility functions.
 * @author kevin
 */
public class FileUtil
{
    /**
     * The logger for this class.
     */
    private final static Logger LOGGER = Logger.getLogger(FileUtil.class) ;
    
    /**
     * Attempt to rename a file 
     * @param from The original file
     * @param to The destination file.
     * @return true if the rename succeeded, false otherwise
     */
    public static boolean renameTo(final File from, final File to)
    {
        if (!from.renameTo(to))
        {
            final File tmpFile ;
            try
            {
                tmpFile = File.createTempFile("copy", ".tmp", to.getParentFile()) ;
            }
            catch (final IOException ioe)
            {
                LOGGER.debug("Could not create temporary file for writing", ioe) ;
                return false ;
            }
            
            try
            {
                copyFile(from, tmpFile) ;
                if (!tmpFile.renameTo(to))
                {
                    LOGGER.debug("Could not rename temporary file " + tmpFile.getAbsolutePath()) ;
                    return false ;
                }
                from.delete();
            }
            finally
            {
                tmpFile.delete() ;
            }
        }
        return true;
    }
    
    /**
     * Attempt to copy the file.
     * @param from The original file
     * @param to The destination file.
     */
    private static boolean copyFile(final File from, final File to)
    {
        final FileInputStream fis;
        try
        {
            fis = new FileInputStream(from);
        }
        catch (final IOException ioe)
        {
            LOGGER.debug("Could not open input file for reading", ioe) ;
            return false ;
        }
        try
        {
            final FileOutputStream fos;
            try
            {
                fos = new FileOutputStream(to);
            }
            catch (final IOException ioe)
            {
                LOGGER.debug("Could not open output file for writing", ioe);
                return false ;
            }
        
            try
            {
                final long filesize = from.length();
                final byte[] buffer = (filesize > 256 ? new byte[256]
                                : new byte[(int) filesize]);
                while (true)
                {
                    final int count = fis.read(buffer);
                    if (count <= 0)
                    {
                        break;
                    }
                    fos.write(buffer, 0, count);
                }
            }
            catch (final IOException ioe)
            {
                LOGGER.debug("Error copying file", ioe);
                to.delete() ;
                return false ;
            }
            finally
            {
                    try
                    {
                            fos.close();
                    }
                    catch (final IOException ioe) {} // ignore
            }
        }
        finally
        {
            try
            {
                    fis.close();
            }
            catch (final IOException ioe) {} // ignore
        }
        return true ;
    }
    
    /**
     * Read the file into a String.
     * @param file - the file to be read
     * @return String with the content of the file
     * @throws IOException - when we can't read the file
     */
    public static String readTextFile(File file) throws IOException 
    {
        StringBuffer sb = new StringBuffer(1024);
        BufferedReader reader = new BufferedReader(new FileReader(file.getPath()));
        char[] chars = new char[1];
        while( (reader.read(chars)) > -1){
            sb.append(String.valueOf(chars)); 
            chars = new char[1];
        }
        reader.close();
        return sb.toString();
    }

    /**
     * Read the contents of a file and return it as a byte array.
     * @param file The file to be read.
     * @return The file contents.
     * @throws IOException Error reading the file.
     */
    public static byte[] readFile(File file) throws IOException {
        ByteArrayOutputStream fileBuffer = new ByteArrayOutputStream();
        InputStream fileInStream = new FileInputStream(file);

        try {
            byte[] readBuffer = new byte[256];
            int readCount = 0;

            while((readCount = fileInStream.read(readBuffer)) != -1) {
                fileBuffer.write(readBuffer, 0, readCount);
            }
        } finally {
            fileInStream.close();
        }

        return fileBuffer.toByteArray();
    }
}
