/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.util;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.beans.PropertyEditor;
import java.beans.PropertyEditorManager;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;

/**
 * The BeanConfigurator knows how to map a ConfigTree to JavaBean properties.
 * 
 * @author <a href="mailto:rex.sheridan@sapience360.com">Rex Sheridan</a>
 */
public class BeanConfigurator {

        static
        {
            try
            {
                ClassUtil.forName("org.jboss.util.propertyeditor.PropertyEditors", BeanConfigurator.class) ;
            }
            catch (final ClassNotFoundException cnfe) {} //ignore
        }
        
	private PropertyDescriptor[] descriptors;

	private Object bean;

	private Map<String, String> properties;

	public BeanConfigurator() {
		super();
	}

	public BeanConfigurator(ConfigTree config, Object bean) {
		List<KeyValuePair> pairs = config.childPropertyList();
		properties = new HashMap<String, String>();
		for (KeyValuePair kvp : pairs) {
			properties.put(kvp.getKey(), kvp.getValue());
		}
		this.bean = bean;
	}

	public BeanConfigurator(Map<String, String> properties, Object bean) {
		this.properties = properties;
		this.bean = bean;
	}

	public void configure() throws IntrospectionException {
		Class<? extends Object> beanClass = bean.getClass();
		BeanInfo beanInfo = Introspector.getBeanInfo(beanClass);
		descriptors = beanInfo.getPropertyDescriptors();

		for (Entry<String, String> entry : properties.entrySet()) {
			String key = entry.getKey();
			key = Introspector.decapitalize(key);
			PropertyDescriptor descriptor = getPropertyDescriptor(key);
			if (descriptor == null) {
				throw new IllegalArgumentException("'" + key
						+ "' property not found on " + beanClass.getName());
			}
			setProperty(descriptor, key, entry.getValue());
		}
	}

	protected void setProperty(PropertyDescriptor descriptor, String name,
			String value) {
		Method setter = descriptor.getWriteMethod();
		if (setter == null) {
			throw new IllegalArgumentException("No setter for property ["
					+ name + "]");
		}

		Object arg;
		Class<?> propertyType = descriptor.getPropertyType();
		PropertyEditor editor = PropertyEditorManager.findEditor(propertyType);
		if (editor == null) {
			throw new IllegalArgumentException(
					"Unable to find suitable property editor for property="
							+ name + " type=" + propertyType.getName());
		}
		editor.setAsText(value);
		arg = editor.getValue();

		try {
			setter.invoke(bean, new Object[] { arg });
		} catch (Exception ex) {
			throw new RuntimeException("Unable to set property named '+" + name
					+ "'", ex);
		}
	}

	protected PropertyDescriptor getPropertyDescriptor(String name) {
		for (int i = 0; i < descriptors.length; i++) {
			if (name.equals(descriptors[i].getName())) {
				return descriptors[i];
			}
		}
		return null;
	}

	/**
	 * @return the bean
	 */
	public Object getBean() {
		return bean;
	}

	/**
	 * @param bean
	 *            the bean to set
	 */
	public void setBean(Object bean) {
		this.bean = bean;
	}

	/**
	 * @return the properties
	 */
	public Map<String, String> getProperties() {
		return properties;
	}

	/**
	 * @param properties
	 *            the properties to set
	 */
	public void setProperties(Map<String, String> properties) {
		this.properties = properties;
	}
}