/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.notification;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.persist.JdbcCleanConn;
import org.jboss.soa.esb.helpers.persist.SimpleDataSource;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;

public class NotifySqlTable extends NotificationTarget
{
	private Logger log = Logger.getLogger( NotifySqlTable.class );
	
	public static final String CHILD_COLUMN = "column";

	public static final String ATT_TABLE = "table";

	public static final String ATT_DATA = "dataColumn";

	public static final String ATT_NAME = "name";

	public static final String ATT_VALUE = "value";

	protected String m_sDriver, m_sURL, m_sUser, m_sPwd, m_sTable, m_sDataCol;

	protected Properties m_oCols;
    private MessagePayloadProxy payloadProxy;

    public NotifySqlTable (ConfigTree configTree) throws ConfigurationException
	{
		super(configTree);
		m_sDriver = getAttr(SimpleDataSource.DRIVER);
		m_sURL = getAttr(SimpleDataSource.URL);
		m_sUser = getAttr(SimpleDataSource.USER);
		m_sPwd = getAttr(SimpleDataSource.PASSWORD);
		m_sTable = getAttr(ATT_TABLE);
		m_sDataCol = getAttr(ATT_DATA);

		m_oCols = new Properties();
		ConfigTree[] oaP = configTree.getChildren(CHILD_COLUMN);
		for (int i1 = 0; i1 < oaP.length; i1++)
		{
			ConfigTree oCurr = oaP[i1];
			String sCol = oCurr.getAttribute(ATT_NAME);
			if (null == sCol)
			{
				continue;
			}
			sCol = sCol.trim();
			if (sCol.length() < 1)
			{
				continue;
			}
			String sVal = oCurr.getAttribute(ATT_VALUE);
			if (null == sVal)
			{
				sVal = "";
			}
			m_oCols.setProperty(sCol, sVal);
		}

        payloadProxy = new MessagePayloadProxy(configTree,
                                               new String[] {BytesBody.BYTES_LOCATION},
                                               new String[] {BytesBody.BYTES_LOCATION});
	} // __________________________________

	private String getAttr (String p_sAtt) throws ConfigurationException
	{
		String sRet = m_oParms.getAttribute(p_sAtt);
		if (null == sRet)
		{
			throw new ConfigurationException("Missing " + p_sAtt + " attribute");
		}
		sRet = sRet.trim();
		if (p_sAtt.equals(SimpleDataSource.PASSWORD))
		{
			return sRet;
		}
		if (sRet.length() < 1)
		{
			throw new ConfigurationException("Empty " + p_sAtt + " attribute");
		}
		return sRet;
	} // __________________________________

	public String getInsertStmt ()
	{
		String[] saCols = new String[m_oCols.size()];
		m_oCols.keySet().toArray(saCols);

		StringBuffer sbCol = new StringBuffer("insert into ").append(m_sTable)
				.append(" ");
		StringBuffer sbPrm = new StringBuffer(" values ");
		for (int i1 = 0; i1 < saCols.length; i1++)
		{
			String sIn = (i1 == 0) ? "(" : ",";
			String sCurrCol = saCols[i1];
			sbCol.append(sIn).append("\"").append(sCurrCol).append("\"");
			sbPrm.append(sIn).append("'").append(m_oCols.getProperty(sCurrCol))
					.append("'");
			// REVIEW: Does the value being inserted not need to be escaped e.g.
			// what if it has a quote in its value??
		}
		return sbCol.append(")").append(sbPrm).append(")").toString();
	} // __________________________________

	public void sendNotification (Message message) throws NotificationException
	{
        Object obj;
        try {
            obj = payloadProxy.getPayload(message);
        } catch (MessageDeliverException e) {
            throw new NotificationException(e);
        }
        String content;
        if (obj instanceof byte[]) {
            content = new String((byte[]) obj);
        } else {
            content = obj.toString();
        }

		m_oCols.setProperty(m_sDataCol, content);
		JdbcCleanConn oConn = null;
		PreparedStatement ps = null;
		try
		{
			oConn = createJdbcCleanConnection();
			ps = oConn.prepareStatement(getInsertStmt());
			oConn.execUpdWait(ps, 3);
			oConn.commit();
		} 
		catch( SQLException e)
		{
			final String errorMsg = "SqlException while trying to notify table. Insert statement : " + getInsertStmt();
			log.error( errorMsg , e );
			throw new NotificationException( errorMsg, e);
		}
		finally
		{
			if ( ps != null )
			{
				try
				{
					ps.close();
				} 
				catch (SQLException e)
				{
					log.error( "SqlException while trying to close prepared statement PS", e );
				}
			}
			if (null != oConn)
			{
				oConn.release();
			}
		}
	} // __________________________________
	
	/*
	 * extracted this method to simplify testing. DanielB
	 */
	protected JdbcCleanConn createJdbcCleanConnection()
	{
		return new JdbcCleanConn(new SimpleDataSource(m_sDriver, m_sURL, m_sUser, m_sPwd));
	}

} // ____________________________________________________________________________
