/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.gateway;

import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.apache.log4j.Logger;
import org.hibernate.CallbackException;
import org.hibernate.EmptyInterceptor;
import org.hibernate.Transaction;
import org.hibernate.type.Type;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.couriers.Courier;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierFactory;
import org.jboss.soa.esb.couriers.CourierUtil;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.ListenerUtil;
import org.jboss.soa.esb.listeners.RegistryUtil;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * HibernateInterceptor intercepts Hibernate events and sends the object they are 
 * being performed upon in a message.     The object is sent in the body of the message
 * with the "hibernateObject" string.   The default composer for the HibernateInterceptor
 * is PackageHibernateMessageContents.
 * 
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 * @since Version 4.2
 * 
 */
public class HibernateInterceptor extends EmptyInterceptor {

	private static final long serialVersionUID = 1L;
	private ArrayList<HibernateEventBean> m_events;
	private final static Logger m_logger = Logger.getLogger(HibernateInterceptor.class);
	private Courier m_courier;
	
	protected Class m_composerClass;
	protected Method m_processMethod;
	protected Object m_composer;
	protected String m_composerName;
	protected ConfigTree m_config;

	protected Collection<EPR> m_targetEprs;
	protected String m_targetServiceCategory, m_targetServiceName;
        
	// Event Strings
	private static final String DELETE_EVENT = "onDelete";
	private static final String SAVE_EVENT = "onSave";
	private static final String LOAD_EVENT = "onLoad";
	private static final String FLUSH_DIRTY_EVENT = "onFlushDirty";
	private static final String COLLECTION_UPDATE_EVENT = "onCollectionUpdate";
	private static final String COLLECTION_REMOVE_EVENT = "onCollectionRemove";
	
	
	/**
	 * This constructor takes in a configuration and a list of hibernate events to 
	 * monitor.
	 * @param f_config
	 * @param f_list
	 * @throws ManagedLifecycleException
	 */
	public HibernateInterceptor(ConfigTree f_config, ArrayList<HibernateEventBean> f_list) throws ManagedLifecycleException {
		m_config = f_config;
		m_events = f_list;
		
        try {
        	m_targetServiceCategory = ListenerUtil.getValue(m_config,
					ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG, null);
			m_targetServiceName = ListenerUtil.getValue(m_config,
					ListenerTagNames.TARGET_SERVICE_NAME_TAG, null);
			
			if (m_targetServiceCategory == null)
				throw new ManagedLifecycleException("No target service category defined!");
			
			if (m_targetServiceName == null)
				throw new ManagedLifecycleException("No target service name defined!");
			
            m_targetEprs = RegistryUtil.getEprs(m_targetServiceCategory,m_targetServiceName);
            if (null == m_targetEprs || m_targetEprs.size() < 1)
                throw new ManagedLifecycleException("EPR <" + m_targetServiceName + "> not found in registry") ;
        } catch (ServiceNotFoundException snfe) {
                throw new ManagedLifecycleException("EPR <" + m_targetServiceName + " "
                        + m_targetServiceName + "> not found in registry");
        } catch (final RegistryException re) {
            throw new ManagedLifecycleException("Unexpected registry exception", re) ;
		} catch (ManagedLifecycleException ex) {
			throw ex;
		}
        
        try {
			resolveComposerClass();
		} catch (ConfigurationException e) {
			// TODO Auto-generated catch block
			throw new ManagedLifecycleException("Problem resolving composer class", e);
		}

	}
	
	/**
	 * Create message and put messageObject in the body.
	 * @param messageObject hibernate object that goes in the body of the message
	 * @return the message just created
	 */
	private Message createMessage(Object messageObject) {
		Object obj = null;
		try {
			obj = m_processMethod.invoke(m_composer, new Object[] { messageObject });
			if (null == obj) {
				m_logger.error("Action class method <" + m_processMethod
						.getName() + "> returned a null object");
			}
		} catch (IllegalArgumentException e) {
			m_logger.error("Error creating message", e);
		} catch (IllegalAccessException e) {
			m_logger.error("Error creating message", e);
		} catch (InvocationTargetException e) {
			m_logger.error("Error creating message", e);
                }
		Message message = (Message) obj;
		return message;
	}
	
	/**
	 * Add the message info to the message properties.
	 * @param message message
	 * @param id id
	 * @param newValues newValues
	 * @param oldValues oldValues
	 * @param propertyNames propertyNames4
	 * @param types types
	 */
	private void addMessageInfo(Message message, Serializable id, Object[] newValues, Object[] oldValues, String[] propertyNames, Type[] types) {
		if (message != null) {
			if (id != null) {
				message.getProperties().setProperty(ListenerTagNames.HIBERNATE_INTERCEPTOR_ID, id);
			}
			message.getProperties().setProperty(ListenerTagNames.HIBERNATE_INTERCEPTOR_NEWVALUES, newValues);
			message.getProperties().setProperty(ListenerTagNames.HIBERNATE_INTERCEPTOR_OLDVALUES, oldValues);
			message.getProperties().setProperty(ListenerTagNames.HIBERNATE_INTERCEPTOR_PROPERTYNAMES, propertyNames);
			message.getProperties().setProperty(ListenerTagNames.HIBERNATE_INTERCEPTOR_TYPES, types);
		}
	}
	
	/** ..
	 * Add the message info to the message properties.
	 * @param message message
	 * @param id id
	 * @param state state
	 * @param propertyNames propertyNames
	 * @param types types
	 */
	private void addMessageInfo(Message message, Serializable id, Object[] state, String[] propertyNames, Type[] types) {
		if (message != null) {
			if (id != null) {
				message.getProperties().setProperty(ListenerTagNames.HIBERNATE_INTERCEPTOR_ID, id);
			}
			message.getProperties().setProperty(ListenerTagNames.HIBERNATE_INTERCEPTOR_STATE, state);
			message.getProperties().setProperty(ListenerTagNames.HIBERNATE_INTERCEPTOR_PROPERTYNAMES, propertyNames);
			message.getProperties().setProperty(ListenerTagNames.HIBERNATE_INTERCEPTOR_TYPES, types);		
		}
	}
	
	/**
	 * Delivers the message.
	 * @param messageObject 
	 */
	private void deliverMessage(Message message) {
		Throwable thrown = null;
		String text = null;

		try {
			boolean bSent = false;
			for (EPR current : m_targetEprs) {
				m_courier = CourierFactory.getCourier(current);

				try {
					if (m_courier.deliver(message)) {
						bSent = true;
                        break;
					}
				} finally {
					CourierUtil.cleanCourier(m_courier) ;
				}
			}
			
			if (!bSent) {
				text = "Target service <" + m_targetServiceCategory + "," + m_targetServiceName + "> is not registered";
				thrown = new Exception(text);
			}
		} catch (ClassCastException e) {
			thrown = e;
			text = "Action class method <" + m_processMethod.getName() + "> returned a non Message object";
		} catch (CourierException e) {
			thrown = e;
			text = "Courier <" + m_courier.getClass().getName() + ".deliverAsync(Message) FAILED";
		} catch (MalformedEPRException ex) {
			thrown = ex;
			text = "Courier <" + m_courier.getClass().getName() + ".deliverAsync(Message) FAILED with malformed EPR.";
		} catch (IllegalArgumentException e) {
			thrown = e;
			text = "Courier <" + m_courier.getClass().getName() + ".deliverAsync(Message) FAILED with IllegalArgumentException.";
		}
		if (null != thrown) {
			m_logger.error(text);
			m_logger.debug(text, thrown);
			//changeStatusToError();
		}
	}
	
	
	/**
	 * This method resolves the composer class.    As a default, the HibernateInterceptor
	 * uses 
	 * @throws ConfigurationException
	 */
	protected void resolveComposerClass() throws ConfigurationException {
		try
		{
            String sProcessMethod = null;
            m_composerName = m_config.getAttribute(ListenerTagNames.GATEWAY_COMPOSER_CLASS_TAG);
            if (null != m_composerName) {
                m_composerClass = ClassUtil.forName(m_composerName, getClass());
                Constructor oConst = m_composerClass.getConstructor(new Class[] { ConfigTree.class });
                m_composer = oConst.newInstance(m_config);
                sProcessMethod = m_config.getAttribute(ListenerTagNames.GATEWAY_COMPOSER_METHOD_TAG, "process");
            } else {
				m_composerName = PackageHibernateMessageContents.class.getName();
				m_composerClass = PackageHibernateMessageContents.class;
				m_composer = new PackageHibernateMessageContents(PackageHibernateMessageContents.createPayloadProxy(m_config));
				sProcessMethod = "process";
				m_logger.debug("No <" + ListenerTagNames.ACTION_ELEMENT_TAG + "> element found in configuration" + " -  Using default composer class : " + m_composerName);
			}
	
			m_processMethod = m_composerClass.getMethod(sProcessMethod, new Class[] { Object.class });
		} catch (InvocationTargetException ex) {
			m_logger.debug(ex);
			throw new ConfigurationException(ex);
		} catch (IllegalAccessException ex) {
			m_logger.debug(ex);
			throw new ConfigurationException(ex);
		} catch (InstantiationException ex) {
			m_logger.debug(ex);	
			throw new ConfigurationException(ex);
		} catch (ClassNotFoundException ex) {
			m_logger.debug(ex);			
			throw new ConfigurationException(ex);
		} catch (NoSuchMethodException ex) {
			m_logger.debug(ex);
			throw new ConfigurationException(ex);
		}
	}
	
	public void afterTransactionBegin(Transaction arg0) {
		m_logger.debug("afterTransactionBegin");
	}

	public void afterTransactionCompletion(Transaction arg0) {
		m_logger.debug("afterTransactionCompletion");
	}

	public void beforeTransactionCompletion(Transaction arg0) {
		m_logger.debug("beforeTransactionCompletion");
	}

	public void onCollectionRecreate(Object entity, Serializable id) throws CallbackException {
		m_logger.debug("onCollectionRecreate");
	}
 
	public void onCollectionRemove(Object entity, Serializable id) throws CallbackException {
		System.out.println("onCollectionRemove");
		for (HibernateEventBean heb: m_events) {
			if (heb.getEvent().equals(COLLECTION_REMOVE_EVENT)) {
				if (entity.getClass().getName().equals(heb.getClassname())) {
					Message message = createMessage(entity);
					message.getProperties().setProperty(ListenerTagNames.HIBERNATE_INTERCEPTOR_ID, id);
					deliverMessage(message);
				}
			}
		}
	}

	public void onCollectionUpdate(Object entity, Serializable id) throws CallbackException {
		System.out.println("onCollectionUpdate");
		for (HibernateEventBean heb: m_events) {
			if (heb.getEvent().equals(COLLECTION_UPDATE_EVENT)) {
				if (entity.getClass().getName().equals(heb.getClassname())) {
					Message message = createMessage(entity);
					message.getProperties().setProperty(ListenerTagNames.HIBERNATE_INTERCEPTOR_ID, id);
					deliverMessage(message);		
				}
			}
		}
	}

	public void onDelete(Object entity, Serializable id, Object[] state, String[] propertyNames, Type[] types) throws CallbackException {
		System.out.println("onDelete");
		for (HibernateEventBean heb: m_events) {
			if (heb.getEvent().equals(DELETE_EVENT)) {
				if (entity.getClass().getName().equals(heb.getClassname())) {
					Message message = createMessage(entity);
					addMessageInfo(message, id, state, propertyNames, types);
					deliverMessage(message);
				}
			}
		}
	}

	public boolean onFlushDirty(Object entity, Serializable id, Object[] newValues, Object[] oldValues, String[] propertyNames, Type[] types) throws CallbackException {
		m_logger.debug("onFlushDirty");
		for (HibernateEventBean heb: m_events) {
			if (heb.getEvent().equals(FLUSH_DIRTY_EVENT)) {
				if (entity.getClass().getName().equals(heb.getClassname())) {
					Message message = createMessage(entity);
					addMessageInfo(message, id, newValues, oldValues, propertyNames, types);
					deliverMessage(message);
				}
			}
		}
		return false;
	}

	public boolean onLoad(Object entity, Serializable id, Object[] state, String[] propertyNames, Type[] types) throws CallbackException {
		System.out.println("onLoad");
		for (HibernateEventBean heb: m_events) {
			if (heb.getEvent().equals(LOAD_EVENT)) {
				if (entity.getClass().getName().equals(heb.getClassname())) {
					Message message = createMessage(entity);
					addMessageInfo(message, id, state, propertyNames, types);
					deliverMessage(message);
				}
			}
		}
		return false;
	}

	public boolean onSave(Object entity, Serializable id, Object[] state,
		String[] propertyNames, Type[] types) throws CallbackException {
		System.out.println("onSave");
		for (HibernateEventBean heb: m_events) {
			if (heb.getEvent().equals(SAVE_EVENT)) {
				if (entity.getClass().getName().equals(heb.getClassname())) {
					Message message = createMessage(entity);
					addMessageInfo(message, id, state, propertyNames, types);
					deliverMessage(message);
				}
			}
		}
		return false;
	}

	public void postFlush(Iterator arg0) throws CallbackException {
		m_logger.debug("postFlush");
	}

	public void preFlush(Iterator arg0) throws CallbackException {
		m_logger.debug("preFlush");
	}	
}
