/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.listeners.config;

import java.io.File;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.URL;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.jboss.deployment.DeploymentException;
import org.jboss.deployment.DeploymentInfo;
import org.jboss.deployment.SubDeployer;
import org.jboss.deployment.SubDeployerSupport;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.logging.Logger;
import org.jboss.metadata.MetaData;
import org.jboss.metadata.XmlFileLoader;
import org.jboss.mx.loading.LoaderRepositoryFactory;
import org.jboss.mx.util.MBeanProxyExt;
import org.jboss.mx.util.ObjectNameConverter;
import org.jboss.soa.esb.lifecycle.LifecycleResourceManager;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.util.XPathNamespaceContext;
import org.jboss.system.ServiceControllerMBean;
import org.jboss.web.AbstractWebContainer;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

/**
 * comment
 *
 * @author <a href="bill@jboss.com">Bill Burke</a>
 * @version $Revision: 1.1 $
 */
public class JBoss4ESBDeployer extends SubDeployerSupport
        implements SubDeployer, JBoss4ESBDeployerMBean
{
   private final static Logger log = Logger.getLogger(JBoss4ESBDeployer.class);

   private ServiceControllerMBean serviceController;
   private Properties actionArtifactProperties;
   
   private static final String PREFIX_CANONICAL_NAME = "jboss.esb:deployment=" ;
   private static final String ESB_ARTIFACT_NAME = "jbossesb.esb" ;
   
   /**
    * Default CTOR used to set default values to the Suffixes and RelativeOrder
    * attributes. Those are read at subdeployer registration time by the MainDeployer
    * to alter its SuffixOrder.
    */
   public JBoss4ESBDeployer()
   {
      setSuffixes(new String[]{".esb"});
      setRelativeOrder(1000); // before old EJB 2.1 deployer
   }

   public static boolean hasFile(DeploymentInfo di, String filePath)
   {
      String urlStr = di.url.getFile();
      try
      {
         URL dd = di.localCl.findResource(filePath);
         if (dd != null)
         {

            // If the DD url is not a subset of the urlStr then this is coming
            // from a jar referenced by the deployment jar manifest and the
            // this deployment jar it should not be treated as persistence
            if (di.localUrl != null)
            {
               urlStr = di.localUrl.toString();
            }

            String ddStr = dd.toString();
            if (ddStr.indexOf(urlStr) >= 0)
            {
               return true;
            }
         }
      }
      catch (Exception ignore)
      {
      }
      return false;
   }

   /**
    * Returns true if this deployer can deploy the given DeploymentInfo.
    *
    * @return True if this deployer can deploy the given DeploymentInfo.
    * @jmx:managed-operation
    */
   public boolean accepts(DeploymentInfo di)
   {
      String urlStr = di.url.toString();
      return urlStr.endsWith(".esb") || urlStr.endsWith(".esb/") ||
              urlStr.endsWith("-esb.xml");
   }

   /**
    * Get a reference to the ServiceController
    */
   protected void startService() throws Exception
   {
      final InputStream actionArtifactIS = ClassUtil.getResourceAsStream("/actionArtifactMap.properties", getClass()) ;
      if (actionArtifactIS == null)
      {
         log.debug("No action artifact mapping") ;
      }
      else
      {
         log.debug("Loading action artifact mapping") ;
         final Properties properties = new Properties() ;
         properties.load(actionArtifactIS) ;
         actionArtifactProperties = properties ;
      }
      
      serviceController = (ServiceControllerMBean)
              MBeanProxyExt.create(ServiceControllerMBean.class,
                      ServiceControllerMBean.OBJECT_NAME, server);

      mainDeployer.addDeployer(this);
      LifecycleResourceManager.deactivateHook() ;
   }
   
   @Override
   protected void stopService() throws Exception
   {
      LifecycleResourceManager.getSingleton().cleanupAllResources() ;
      super.stopService();
   }


   protected URL getDocumentUrl(DeploymentInfo di)
   {
      String urlStr = di.url.toString();
      if (urlStr.endsWith(".esb") || urlStr.endsWith(".esb/"))
      {
         return di.localCl.getResource("META-INF/jboss-esb.xml");
      }
      return di.url;
   }

   public void init(DeploymentInfo di) throws DeploymentException
   {
      try
      {
         if (di.url.getProtocol().equalsIgnoreCase("file"))
         {
            File file = new File(di.url.getFile());

            if (!file.isDirectory())
            {
               // If not directory we watch the package
               di.watch = di.url;
            }
            else
            {
               // If directory we watch the xml files
               di.watch = new URL(di.url, "META-INF/jboss-esb.xml");
            }
         }
         else
         {
            // We watch the top only, no directory support
            di.watch = di.url;
         }

         XmlFileLoader xfl = new XmlFileLoader();
         InputStream in = di.localCl.getResourceAsStream("META-INF/deployment.xml");
         if (in != null)
         {
            try
            {
               Element jboss = xfl.getDocument(in, "META-INF/deployment.xml").getDocumentElement();
               // Check for a ejb level class loading config
               Element loader = MetaData.getOptionalChild(jboss, "loader-repository");
               if (loader != null)
               {
                  LoaderRepositoryFactory.LoaderRepositoryConfig config =
                          LoaderRepositoryFactory.parseRepositoryConfig(loader);
                  di.setRepositoryInfo(config);
               }

            }
            finally
            {
               in.close();
            }
         }
      }
      catch (Exception e)
      {
         if (e instanceof DeploymentException)
         {
            throw(DeploymentException) e;
         }
         throw new DeploymentException("failed to initialize", e);
      }

      // invoke super-class initialization
      super.init(di);
   }

   public synchronized void create(DeploymentInfo di) throws DeploymentException
   {
      log.info("create esb service, " + di.shortName);
      URL document = getDocumentUrl(di);
      
      if (document == null)
      {
         throw new DeploymentException("Unable to find document url of META-INF/jboss-esb.xml in: "
                 + di.url);
      }
      try
      {
         Set<ObjectName> deps = new HashSet<ObjectName>();
         InputStream in = di.localCl.getResourceAsStream("META-INF/deployment.xml");
         if (in != null)
         {
            try
            {
               XmlFileLoader xfl = new XmlFileLoader();
               Element jboss = xfl.getDocument(in, "META-INF/deployment.xml").getDocumentElement();
               // Check for a ejb level class loading config
               Iterator depends = MetaData.getChildrenByTagName(jboss, "depends");
               if (depends != null)
               {
                  while (depends.hasNext())
                  {
                     Element depend = (Element)depends.next();
                     ObjectName depOn = new ObjectName(MetaData.getElementContent(depend));
                     deps.add(depOn);
                  }
               }
               Iterator esbDepends = MetaData.getChildrenByTagName(jboss, "esb-depends");
               if ((esbDepends != null) && esbDepends.hasNext())
               {
                  final Map<String, DeploymentInfo> subDeploymentLocationMap ;
                  if (di.subDeployments.size() > 0)
                  {
                     subDeploymentLocationMap = new HashMap<String, DeploymentInfo>() ;
                     final Set<DeploymentInfo> subDeployments = (Set<DeploymentInfo>)di.subDeployments ;
                     for(DeploymentInfo subDI : subDeployments)
                     {
                        final String urlPath = subDI.url.getPath() ;
                        final String deployablePath = (urlPath.endsWith("/") ? urlPath.substring(0, urlPath.length()-1) : urlPath) ;
                        final int lastSeparator = deployablePath.lastIndexOf('/') ;
                        final String deployable = (lastSeparator >= 0 ? deployablePath.substring(lastSeparator+1) : deployablePath) ;
                        if (subDeploymentLocationMap.put(deployable, subDI) != null)
                        {
                           throw new DeploymentException("Duplicate subDeployment name: " + deployable) ;
                        }
                     }
                  }
                  else
                  {
                     throw new DeploymentException("No subdeployments to match esb-depends") ;
                  }
                  
                  do
                  {
                     Element depend = (Element)esbDepends.next();
                     final String deployable = MetaData.getElementContent(depend) ;
                     final DeploymentInfo subDI = subDeploymentLocationMap.get(deployable) ;
                     if ((subDI != null) && subDI.context.containsKey(AbstractWebContainer.WEB_MODULE))
                     {
                        final ObjectName jmxName = (ObjectName) subDI.context.get(AbstractWebContainer.WEB_MODULE) ;
                        deps.add(jmxName) ;
                     }
                     else
                     {
                         throw new DeploymentException("Could not locate WAR subdeployment matching: " + deployable) ;
                     }
                  }
                  while (esbDepends.hasNext()) ;
               }
            }
            finally
            {
               in.close();
            }
         }

         final InputStream inputStream = document.openStream();
         final String jbossEsbXml ;
         try
         {
             final StringWriter sw = new StringWriter() ;
             XMLHelper.replaceSystemProperties(XMLHelper.getXMLStreamReader(inputStream),
                 XMLHelper.getXMLStreamWriter(sw)) ;
             jbossEsbXml = sw.toString();
         }
         finally
         {
             inputStream.close();
         }
         
         addActionDependencies(di.shortName, jbossEsbXml, deps) ;
         
         JBoss4ESBDeployment deployment = new JBoss4ESBDeployment(jbossEsbXml, di.shortName);
         deployment.setClassloader(di.ucl);
         String name = PREFIX_CANONICAL_NAME + di.shortName;
         ObjectName on = ObjectNameConverter.convert(name);
         // Check that the name is not registered
         if (server.isRegistered(on) == true)
         {
            log.debug("The ESBModule name: " + name
                      + "is already registered, adding uid=" + System.identityHashCode(deployment));
            name = name + ",uid=" + System.identityHashCode(deployment);
            on = ObjectNameConverter.convert(name);
         }
         server.registerMBean(deployment, on);
         di.deployedObject = on;
         log.debug("Deploying: " + di.url);
         // Invoke the create life cycle method
         serviceController.create(di.deployedObject, deps);
      }
      catch (Exception e)
      {
         throw new DeploymentException("Error during create of ESB Module: "
                 + di.url, e);
      }
      super.create(di);
   }

   private void addActionDependencies(final String deploymentName,
      final String configuration, final Set<ObjectName> deps)
      throws XPathExpressionException, DeploymentException, MalformedObjectNameException
   {
      final XmlFileLoader xfl = new XmlFileLoader() ;
      final StringReader sr = new StringReader(configuration) ;
      final InputSource is = new InputSource(sr) ;
      final Element rootElement = xfl.getDocument(is, "META-INF/jboss-esb.xml").getDocumentElement() ;

      final Set<String> artifacts = new HashSet<String>() ;
      artifacts.add(ESB_ARTIFACT_NAME) ;

      final String namespaceURI = "http://anonsvn.labs.jboss.com/labs/jbossesb/trunk/product/etc/schemas/xml/jbossesb-1.0.1.xsd" ;
      
      final XPath xpathEvaluater = XPathFactory.newInstance().newXPath();
      final XPathNamespaceContext namespaceContext = new XPathNamespaceContext() ;
      namespaceContext.setMapping("jbesb", namespaceURI) ;
      xpathEvaluater.setNamespaceContext(namespaceContext) ;

      final NodeList actionList = (NodeList) xpathEvaluater.evaluate("/jbesb:jbossesb/jbesb:services/jbesb:service/jbesb:actions/jbesb:action",
              rootElement, XPathConstants.NODESET) ; 
      final int numActions = actionList.getLength() ;
      if (numActions > 0)
      {
          final Set<String> actionClasses = new HashSet<String>() ;
          for(int count = 0 ; count < numActions ; count++)
          {
              final Node actionNode = actionList.item(count) ;
              if (!(actionNode instanceof Element))
              {
                  log.warn("Action node is not an element: " + actionNode) ;
              }
              else
              {
                  final Element actionElement = (Element)actionNode ;
                  final String actionClass = actionElement.getAttribute("class") ;
                  actionClasses.add(actionClass) ;
              }
          }
          
          final int numActionClasses = actionClasses.size() ;
          if (numActionClasses > 0)
          {
              for(final String actionClass: actionClasses)
              {
                  final String artifact = (String)actionArtifactProperties.get(actionClass) ;
                  if (artifact != null)
                  {
                      artifacts.add(artifact) ;
                  }
              }
          }
      }
      
      for(final String artifact: artifacts)
      {
          if (!deploymentName.equals(artifact))
          {
              final String canonicalName = PREFIX_CANONICAL_NAME + artifact ;
              final ObjectName on = ObjectNameConverter.convert(canonicalName) ;
              deps.add(on) ;
          }
      }
   }

   public synchronized void start(DeploymentInfo di)
           throws DeploymentException
   {
      try
      {
         serviceController.start(di.deployedObject);
      }
      catch (Exception e)
      {
         try
         {
            stop(di);
            destroy(di);
         }
         catch (DeploymentException ignore)
         {
         }
         throw new DeploymentException("Error during start of ESB Module: "
                 + di.url, e);
      }

      super.start(di);
   }

   public void stop(DeploymentInfo di)
           throws DeploymentException
   {
      if (di.deployedObject != null)
      {
         try
         {
            serviceController.stop(di.deployedObject);
         }
         catch (Exception e)
         {
            throw new DeploymentException("Error during stop of ESB Module: "
                    + di.url, e);
         }
      }
      super.stop(di);
   }

   public void destroy(DeploymentInfo di)
           throws DeploymentException
   {
      if (di.deployedObject != null)
      {
         try
         {
            serviceController.destroy(di.deployedObject);
            server.unregisterMBean(di.deployedObject);
         }
         catch (Exception e)
         {
            throw new DeploymentException("Error during stop of ESB Module: "
                    + di.url, e);
         }
      }
      super.destroy(di);
   }
}
