/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.http.configurators;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.contrib.ssl.StrictSSLProtocolSocketFactory;
import org.apache.commons.httpclient.protocol.ProtocolSocketFactory;
import org.apache.commons.httpclient.protocol.DefaultProtocolSocketFactory;
import org.apache.commons.httpclient.protocol.Protocol;
import org.apache.commons.ssl.KeyMaterial;
import org.apache.commons.ssl.SSLClient;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.http.Configurator;
import org.jboss.soa.esb.util.ClassUtil;

import java.util.Properties;
import java.net.URI;
import java.net.URISyntaxException;
import java.io.*;
import java.security.GeneralSecurityException;
import java.security.NoSuchAlgorithmException;
import java.security.KeyStoreException;
import java.security.KeyManagementException;
import java.security.cert.CertificateException;

/**
 * HTTP Protocol configurator.
 * <p/>
 * Supports both HTTP and HTTPS, including configuring the socket factory and SSL keystore.  It
 * supports the SSL keystore being on the classpath, filesystem or based on a {@link URI}.
 * <p/>
 * Properties:
 * <ul>
 *      <li><b>target-host-url</b>: (Required)</li>
 *      <li><b>keystore</b>: (Optional).  See {@link KeyMaterial}. Defaults to "/keystore".</li>
 *      <li><b>keystore-passwd</b>: (Optional).  See {@link KeyMaterial}. Defaults to "changeit".</li>
 *      <li><b>protocol-socket-factory</b>: See {@link Protocol}. (Optional).
 *          Defaults to {@link StrictSSLProtocolSocketFactory} for HTTPS, otherwise defaults to {@link DefaultProtocolSocketFactory}.
 *          Configure with {@link org.apache.commons.httpclient.contrib.ssl.EasySSLProtocolSocketFactory} if you wish
 *          to allow the target server to authenticate with a self-signed certificate</li>
 * </ul>
 * <p/>
 * See <a href="http://jakarta.apache.org/commons/httpclient/sslguide.html">HttpClient HttpProtocol Guide</a>.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class HttpProtocol extends Configurator {

    public void configure(HttpClient httpClient, Properties properties) throws ConfigurationException {
        URI targetURI = getTargetURI(properties, true);
        String factory;
        String scheme = targetURI.getScheme();
        int port = targetURI.getPort();
        org.apache.commons.httpclient.protocol.Protocol protocol;
        KeyMaterial keyMaterial = null;
        ProtocolSocketFactory socketFactory;

        if(!scheme.startsWith("http")) {
            // We're only interested in HTTP for this...
            return;
        }

        if(scheme.equals("https")) {
            factory = properties.getProperty("protocol-socket-factory", StrictSSLProtocolSocketFactory.class.getName());
            keyMaterial = getKeyMaterial(properties);

            if(port == -1) {
                port = 443;
            }
        } else {
            factory = properties.getProperty("protocol-socket-factory", DefaultProtocolSocketFactory.class.getName());
            if(port == -1) {
                port = 80;
            }
        }
        assertPropertySetAndNotBlank(factory, "protocol-socket-factory");
        socketFactory = createFactoryClass(factory, keyMaterial);

        // And finally... configure the host with the protocol....
        protocol = new Protocol(scheme, socketFactory, port);
        Protocol.registerProtocol(scheme, protocol);
        //httpClient.getHostConfiguration().setHost(targetURI.getHost(), port, protocol);
    }

    private KeyMaterial getKeyMaterial(Properties properties) throws ConfigurationException {
        String keyStore = properties.getProperty("keystore", "/keystore");
        String keyStorePassword = properties.getProperty("keystore-passw", "changeit");

        // Try it as a classpath resource ...
        InputStream keyStoreStream = ClassUtil.getResourceAsStream(keyStore, HttpProtocol.class);

        try {
            // Try it as a File resource...
            if(keyStoreStream == null) {
                File keyStoreFile = new File(keyStore);

                if(keyStoreFile.exists() && !keyStoreFile.isDirectory()) {
                    return new KeyMaterial(new FileInputStream(keyStoreFile), keyStorePassword.toCharArray());
                }
            } else {
                return new KeyMaterial(keyStoreStream, keyStorePassword.toCharArray());
            }

            // Try it as a URI resource...
            if(keyStoreStream == null) {
                try {
                    URI fileURI = new URI(keyStore);                    
                    if(fileURI.isAbsolute()) {
                        return new KeyMaterial(fileURI.toURL().openStream(), keyStorePassword.toCharArray());
                    }
                } catch (URISyntaxException e) {
                    throw new ConfigurationException("Failed to load keystore '" + keyStore + "'.");
                }
            }
        } catch (IOException e) {
            throw new ConfigurationException("Failed to load keystore '" + keyStore + "'.", e);
        } catch (GeneralSecurityException e) {
            throw new ConfigurationException("Failed to load keystore '" + keyStore + "'.", e);
        }

        throw new ConfigurationException("Failed to locate keystore '" + keyStore + "'.");
    }

    private ProtocolSocketFactory createFactoryClass(String factory, KeyMaterial keyMaterial) throws ConfigurationException {
        ProtocolSocketFactory socketFactory = null;

        try {
            socketFactory = (ProtocolSocketFactory) ClassUtil.forName(factory, HttpProtocol.class).newInstance();
        } catch (ClassCastException e) {
            throw new ConfigurationException("Class [" + factory + "] must implement [" + ProtocolSocketFactory.class.getName() + "].", e);
        } catch (ClassNotFoundException e) {
            throw new ConfigurationException("ProtocolSocketFactory implementation class [" + factory + "] not found in classpath.", e);
        } catch (InstantiationException e) {
            throw new ConfigurationException("Failed to instantiate ProtocolSocketFactory implementation class [" + factory + "].", e);
        } catch (IllegalAccessException e) {
            throw new ConfigurationException("Failed to instantiate ProtocolSocketFactory implementation class [" + factory + "].", e);
        }

        if(socketFactory instanceof SSLClient && keyMaterial != null) {
            try {
                ((SSLClient)socketFactory).setKeyMaterial(keyMaterial);
            } catch (NoSuchAlgorithmException e) {
                throw new ConfigurationException("Failed to configure SSL Keystore on SSLClient.", e);
            } catch (KeyStoreException e) {
                throw new ConfigurationException("Failed to configure SSL Keystore on SSLClient.", e);
            } catch (KeyManagementException e) {
                throw new ConfigurationException("Failed to configure SSL Keystore on SSLClient.", e);
            } catch (IOException e) {
                throw new ConfigurationException("Failed to configure SSL Keystore on SSLClient.", e);
            } catch (CertificateException e) {
                throw new ConfigurationException("Failed to configure SSL Keystore on SSLClient.", e);
            }
        }

        return socketFactory;
    }


}
