/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.helpers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;
import java.util.Map.Entry;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NameNotFoundException;
import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.lifecycle.LifecyclePriorities;
import org.jboss.soa.esb.lifecycle.LifecycleResource;
import org.jboss.soa.esb.lifecycle.LifecycleResourceException;
import org.jboss.soa.esb.lifecycle.LifecycleResourceFactory;

import com.arjuna.common.util.propertyservice.PropertyManager;

/**
 * Pool for Naming Contexts.
 * 
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
public class NamingContextPool
{
    /**
     * Logger for this class
     */
    private static final Logger LOGGER = Logger.getLogger(NamingContextPool.class);
    /**
     * Default maximum pool size.
     */
    private static final int DEFAULT_POOL_SIZE = 20 ;
    /**
     * Default sleep period.
     */
    private static final int DEFAULT_SLEEP_PERIOD = 30 ;
    /**
     * Default retry count for creating Naming Contexts.
     */
    private static final int DEFAULT_RETRY_COUNT = 10 ;
    /**
     * Default key for environments.
     */
    private static final String DEFAULT_KEY = "<empty key>" ;
    /**
     * Maximum pool size.
     */
    private static final int POOL_SIZE ;
    /**
     * Sleep period.
     */
    private static final int SLEEP_PERIOD ;
    /**
     * Retry count for creating Naming Contexts.
     */
    private static final int RETRY_COUNT ;

    /**
     * The lifecycle resource factory.
     */
    private static final LifecycleResourceFactory<NamingContextPool> lifecycleNamingContextPoolFactory =
        new NamingContextPoolFactory();

    /**
     * The lifecycle Naming Context pools.
     */
    private static final LifecycleResource<NamingContextPool> lifecycleNamingContextPoolResource =
        new LifecycleResource<NamingContextPool>(lifecycleNamingContextPoolFactory,
            LifecyclePriorities.NAMING_CONTEXT_POOL_PRIORITY);
    
    /**
     * The free contexts.
     */
    private final Map<String, List<Context>> freeContexts = new HashMap<String, List<Context>>() ;
    /**
     * The free contexts in insertion order.
     */
    private final Map<Context, String> freeContextOrder = new LinkedHashMap<Context, String>() ;
    /**
     * The in use contexts.
     */
    private final Map<Context, String> inUseContexts = new HashMap<Context, String>() ;
    /**
     * The count of created contexts.
     */
    private int numContexts ;
    /**
     * True if the pool has been destroyed.
     */
    private boolean destroyed ;
    
    /**
     * Private constructor.
     */
    private NamingContextPool()
    {
    }
    
    /**
     * Get a Naming Context from the pool.
     * @param env The JNDI environment parameters for the Naming Context.
     * @return The Naming Context.
     * @throws NamingContextException for errors obtaining a Naming Context.
     */
    Context getContext(final Properties env)
        throws NamingContextException
    {
        final long end = System.currentTimeMillis() + (SLEEP_PERIOD * 1000) ;
        final String key = getKey(env) ;
        boolean emitExpiry = LOGGER.isDebugEnabled() ;
        synchronized(this)
        {
            while(true)
            {
                if (destroyed)
                {
                    throw new NamingContextException("Naming Context Pool has been previosuly destroyed") ;
                }
                final Context context = getContext(env, key) ;
                if (context != null)
                {
                    return context ;
                }
                if (emitExpiry)
                {
                    LOGGER.debug("The Naming Context pool was exhausted, waiting for one to be released.") ;
                    emitExpiry = false ;
                }
                final long now = System.currentTimeMillis() ;
                final long delay = (end - now) ;
                if (delay <= 0)
                {
                    throw new NamingContextException("Could not obtain a Naming Context from the pool after " + SLEEP_PERIOD + "s.") ;
                }
                else
                {
                    try
                    {
                        wait(delay) ;
                    }
                    catch (final InterruptedException ie) {} // ignore
                }
            }
        }
    }
    
    /**
     * Release the Naming Context back into the pool.
     * @param context The Naming Context.
     * @throws NamingContextException for errors releasing the Naming Context.
     */
    synchronized void releaseContext(final Context context)
        throws NamingContextException
    {
        if (destroyed)
        {
            throw new NamingContextException("Naming Context Pool has been previosuly destroyed") ;
        }
        final String key = inUseContexts.remove(context) ;
        if (key == null)
        {
            throw new NamingContextException("Cannot release a context which is not in use.") ;
        }
        
        freeContextOrder.put(context, key) ;
        final List<Context> contexts = freeContexts.get(key) ;
        if (contexts != null)
        {
            contexts.add(context) ;
        }
        else
        {
            final List<Context> newContexts = new ArrayList<Context>() ;
            newContexts.add(context) ;
            freeContexts.put(key, newContexts) ;
        }
        
        notifyAll() ;
    }
    
    /**
     * Replace the specified Naming Context.
     * @param context The Naming Context to replace.
     * @param env The JNDI environment parameters for the Naming Context.
     * @return The new Naming Context
     * @throws NamingContextException for errors releasing the Naming Context.
     */
    synchronized Context replaceContext(final Context context, final Properties env)
        throws NamingContextException
    {
        if (destroyed)
        {
            throw new NamingContextException("Naming Context Pool has been previosuly destroyed") ;
        }
        final String key = inUseContexts.remove(context) ;
        if (key == null)
        {
            throw new NamingContextException("Cannot release a context which is not in use.") ;
        }
        
        numContexts-- ;
        closeContext(context) ;
        return createContext(env, key) ;
    }

    /**
     * Closes all Contexts, and removes them from the contextCache.
     */
    synchronized void closeAllContexts()
    {
        closeAllContexts(freeContextOrder.keySet().iterator()) ;
        if (inUseContexts.size() > 0)
        {
            LOGGER.warn("Forcing closure of in-use Naming Contexts") ;
            closeAllContexts(inUseContexts.keySet().iterator()) ;
        }
        inUseContexts.clear() ;
        freeContextOrder.clear() ;
        freeContexts.clear() ;
        destroyed = true ;
    }
    
    /**
     * Get a Naming Context from the pool.
     * @param env The JNDI environment parameters for the Naming Context.
     * @param key The key of the context to locate.
     * @return The Naming Context.
     * @throws NamingContextException for errors obtaining a Naming Context.
     */
    private Context getContext(final Properties env, final String key)
        throws NamingContextException
    {
        final Context context = getFreeContext(key) ;
        if (context != null)
        {
            return context ;
        }
        if (numContexts == POOL_SIZE)
        {
            if (freeContextOrder.size() == 0)
            {
                return null ;
            }
            final Iterator<Entry<Context, String>> freeContextIter = freeContextOrder.entrySet().iterator() ;
            final Entry<Context, String> freeContextEntry = freeContextIter.next() ;
            final Context freeContext = freeContextEntry.getKey() ;
            final String freeContextKey = freeContextEntry.getValue() ;
            if (LOGGER.isDebugEnabled())
            {
                LOGGER.debug("Ejecting Naming Context from pool, key: " + freeContextKey) ;
            }
            
            freeContextOrder.remove(freeContext) ;
            final List<Context> contexts = freeContexts.get(key) ;
            contexts.remove(freeContext) ;
            if (contexts.size() == 0)
            {
                freeContexts.remove(key) ;
            }
            
            numContexts-- ;
            closeContext(freeContext) ;
        }
        return createContext(env, key) ;
    }
    
    /**
     * Get a free Naming Context from the available list
     * @param key The key of the context to return.
     * @return The Naming Context or null if none present
     */
    private Context getFreeContext(final String key)
    {
        final List<Context> contexts = freeContexts.get(key) ;
        if (contexts != null)
        {
            final int size = contexts.size() ;
            if (size > 0)
            {
                final Context context = contexts.remove(size-1) ;
                freeContextOrder.remove(context) ;
                inUseContexts.put(context, key) ;
                return context ;
            }
        }
        return null ;
    }
    
    /**
     * Create the context with the specified environment and key.
     * @param env The JNDI environment parameters for the Naming Context.
     * @param key The key of the context to locate.
     * @return The Naming Context.
     * @throws NamingContextException for errors obtaining a Naming Context.
     */
    private Context createContext(final Properties env, final String key)
        throws NamingContextException
    {
        NamingException cachedException = null ;
        
        for(int count = 0 ; count < RETRY_COUNT ; count++)
        {
            final Context context ;
            try
            {
                context = new InitialContext(env) ;
                
                try
                {
                    context.list("__dummy2@#$%") ;
                }
                catch (final NameNotFoundException nfne) {} // Expected
            }
            catch (final NamingException ne)
            {
                cachedException = ne ;
                continue ;
            }
            
            inUseContexts.put(context, key) ;
            numContexts++ ;
            return context ;
        }
        
        throw new NamingContextException("Failed to create Naming Context", cachedException) ;
    }
    
    /**
     * Close all the specified contexts.
     * @param contextIter The iterator of Naming Contexts.
     */
    private void closeAllContexts(Iterator<Context> contextIter)
    {
        while(contextIter.hasNext())
        {
            closeContext(contextIter.next()) ;
        }
    }
    
    /**
     * Close the specified Naming Context.
     * @param context The Naming Context to close.
     */
    private void closeContext(final Context context)
    {
        try
        {
            context.close() ;
        }
        catch (final NamingException ne) {} // ignore
    }
    
    static
    {
        final PropertyManager prop = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE);
        POOL_SIZE = getIntProperty(prop, Environment.NAMING_CONTEXT_POOL_SIZE, DEFAULT_POOL_SIZE);
        SLEEP_PERIOD = getIntProperty(prop, Environment.NAMING_CONTEXT_SLEEP_PERIOD, DEFAULT_SLEEP_PERIOD);
        RETRY_COUNT = getIntProperty(prop, Environment.NAMING_CONTEXT_RETRY_COUNT, DEFAULT_RETRY_COUNT);
    }
    
    /**
     * Get a Naming Context from the pool.
     * @param properties The properties of the JNDI environment.
     * @return the pooled Naming Context.
     */
    public static Context getNamingContext(final Properties properties)
        throws NamingContextException
    {
        return getPool().getContext(properties) ;
    }
    
    /**
     * Release the Naming Context back into the pool.
     * @param context The Naming Context to release.
     */
    public static void releaseNamingContext(final Context context)
        throws NamingContextException
    {
        getPool().releaseContext(context) ;
    }
    
    /**
     * Close the Naming Context and remove from the pool.
     * @param context The Naming Context to close.
     * @param env The JNDI environment parameters for the Naming Context.
     */
    public static Context replaceNamingContext(final Context context, final Properties env)
        throws NamingContextException
    {
        return getPool().replaceContext(context, env) ;
    }
    
    /**
     * Get the Naming Context pool.
     * @return The Naming Context pool.
     * @throws NamingContextException For errors retrieving the Naming Context pool.
     */
    private static NamingContextPool getPool()
        throws NamingContextException
    {
        try
        {
            return lifecycleNamingContextPoolResource.getLifecycleResource() ;
        }
        catch (final LifecycleResourceException lre)
        {
            throw new NamingContextException("Unexpected lifecycle resource exception", lre) ;
        }
    }
    
    /**
     * Takes properties and serializes this into a long string which is the key
     * into the contextCache.
     * 
     * @param properties -
     *                property Bag.
     * @return key into the contextCache.
     */
    private static String getKey(Properties properties)
    {
        if ((properties == null) || (properties.size() == 0))
        {
            return DEFAULT_KEY ;
        }
        
        final TreeMap<String, String> orderedProperties = new TreeMap<String, String>() ;
        final Iterator<Entry<Object, Object>> entryIter = properties.entrySet().iterator() ;
        while(entryIter.hasNext())
        {
            final Entry<Object, Object> entry = entryIter.next() ;
            orderedProperties.put(entry.getKey().toString(), entry.getValue().toString()) ;
        }
        return orderedProperties.toString() ;
    }
    
    /**
     * Get the integer value of the specified property.
     * @param prop The property manager.
     * @param name The property name.
     * @param defaultValue The default value.
     * @return The integer value of the property or the default value.
     */
    private static int getIntProperty(final PropertyManager prop, final String name, final int defaultValue)
    {
        final String value = prop.getProperty(name) ;
        if (value != null)
        {
            try
            {
                return Integer.parseInt(value) ;
            }
            catch (final NumberFormatException nfe)
            {
                LOGGER.warn("Could not parse value for property: " + name + ", value: " + value) ;
            }
        }
        return defaultValue ;
    }
    
    /**
     * The factory for creating and destroying lifecycle resources.
     */
    private static class NamingContextPoolFactory implements LifecycleResourceFactory<NamingContextPool>
    {
        /**
         * Create a resource object which will be associated with the specified lifecycle identity.
         * @param lifecycleIdentity The associated lifecycle identity.
         * @return The lifecycle resource
         * @throws LifecycleResourceException for errors during construction.
         */
        public NamingContextPool createLifecycleResource(final String lifecycleIdentity)
            throws LifecycleResourceException
        {
            return new NamingContextPool() ;
        }
        
        /**
         * Destroy a resource object which is associated with the specified lifecycle identity.
         * @param resource The lifecycle resource.
         * @param lifecycleIdentity The associated lifecycle identity.
         * @return The lifecycle resource.
         * @throws LifecycleResourceException for errors during destroy.
         */
        public void destroyLifecycleResource(final NamingContextPool resource,
            final String lifecycleIdentity)
            throws LifecycleResourceException
        {
            resource.closeAllContexts() ;
        }
    }
}
