/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.filter;

import java.util.Collection;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.util.ClassUtil;

import com.arjuna.common.util.propertyservice.PropertyManager;

/**
 * The FilterManager process messages through the various input and output
 * filters that have been registered via the configuration file. Currently this
 * is the same processing for all services/clients, i.e., there is no
 * per-process/per-client processing, it is all at the ESB instance level.
 * 
 * @author marklittle
 */

public class FilterManager
{
    public static final synchronized FilterManager getInstance ()
    {
	if (_instance == null)
	    _instance = new FilterManager();

	return _instance;
    }

    /**
         * Iterate through the registered output filters, allowing them to
         * augment/modify the message. The first error causes the iteration to
         * stop and the send to fail.
         * 
         * @param msg
         *                the original message.
         * @return the resultant message.
         * @throws CourierException
         *                 thrown if there is any error during processing.
         */

    public Message doOutputWork (Message msg, Map<String, Object> params)
	    throws CourierException
    {
	if ((_filters == null) || (msg == null))
	    return msg;

	for (int i = 0; i < _filters.length; i++)
	{
	    if (_filters[i] != null)
		msg = _filters[i].onOutput(msg, params);
	}

	return msg;
    }

    /**
         * Iterate through the registered input filters, allowing them to
         * augment/modify the message. The first error causes the iteration to
         * stop and the receive to fail.
         * 
         * @param msg
         *                the original message.
         * @return the resultant message.
         * @throws CourierException
         *                 thrown if there is any error during processing.
         */

    public Message doInputWork (Message msg, Map<String, Object> params)
	    throws CourierException
    {
	if ((_filters == null) || (msg == null))
	    return msg;

	for (int i = 0; i < _filters.length; i++)
	{
	    if (_filters[i] != null)
		msg = _filters[i].onInput(msg, params);
	}

	return msg;
    }

    private FilterManager()
    {
	PropertyManager pm = ModulePropertyManager
		.getPropertyManager(ModulePropertyManager.FILTER_MODULE);
	Properties props = pm.getProperties();

	if (props != null)
	{
	    Enumeration names = props.propertyNames();
	    TreeMap<Integer, String> map = new TreeMap<Integer, String>();

	    /*
                 * Go through the list of attributes and pull out those that
                 * refer to filters. Make sure we order them according to their
                 * names.
                 */

	    while (names.hasMoreElements())
	    {
		String attrName = (String) names.nextElement();

		if (attrName.startsWith(Environment.FILTER_NAME))
		{
		    String order = attrName.substring(Environment.FILTER_NAME
			    .length() + 1);
		    Integer value;

		    try
		    {
			value = new Integer(order);
		    }
		    catch (NumberFormatException ex)
		    {
			throw new RuntimeException("Filter name " + order
				+ " is invalid!");
		    }

		    map.put(value, props.getProperty(attrName));
		}
	    }

	    if (map.size() > 0)
	    {
		Collection<String> ordered = map.values();
		Iterator<String> iter = ordered.iterator();
		int index = 0;

		_filters = new InputOutputFilter[map.size()];

		while (iter.hasNext())
		{
		    String filterName = iter.next();

		    try
		    {
			Class c = ClassUtil.forName(filterName, getClass());
			InputOutputFilter theFilter = (InputOutputFilter) c
				.newInstance();

			_filters[index++] = theFilter;
		    }
		    catch (ClassNotFoundException ex)
		    {
			_logger.warn("FilterManager problem loading class "+filterName, ex);
		    }
		    catch (Throwable ex)
		    {
			_logger.warn("FilterManager problem during load "+filterName, ex);
		    }
		}
	    }
	}
    }

    private InputOutputFilter[] _filters;

    // It's public so all filters can use it if necessary.
    
    public final static Logger _logger = Logger.getLogger(FilterManager.class);

    private static FilterManager _instance = null;
}
