/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.common;

import java.util.Properties;

import javax.jms.ConnectionFactory;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.helpers.NamingContextException;
import org.jboss.soa.esb.helpers.NamingContextPool;

public class Configuration
{
    private static Logger _logger = Logger.getLogger(Configuration.class);
    
	private static KeyValuePair[] s_oaKV = new KeyValuePair[]
	{
			new KeyValuePair(Environment.SMTP_HOST, 					getSmtpHost()),
			new KeyValuePair(Environment.SMTP_USERNAME, 				getSmtpUsername()),
			new KeyValuePair(Environment.SMTP_PASSWORD, 				getSmtpPassword()),
			new KeyValuePair(Environment.SMTP_PORT, 					getSmtpPort()),
			new KeyValuePair(Environment.SMTP_FROM, 					getSmtpFrom()),
			new KeyValuePair(Environment.SMTP_AUTH, 					getSmtpAuth()),
			new KeyValuePair(Environment.JNDI_SERVER_URL, 				getJndiServerURL()),
			new KeyValuePair(Environment.JNDI_SERVER_CONTEXT_FACTORY,   getJndiServerContextFactory()),
			new KeyValuePair(Environment.JNDI_SERVER_PKG_PREFIX, 	    getJndiServerPkgPrefix()),
			new KeyValuePair(Environment.REGISTRY_QUERY_MANAGER_URI, 	getRegistryQueryManageURI()),
			new KeyValuePair(Environment.REGISTRY_LIFECYCLE_MANAGER_URI,getRegistryLifecycleManagerURI()),
			new KeyValuePair(Environment.REGISTRY_IMPEMENTATION_CLASS,  getRegistryImplementationClass()),
			new KeyValuePair(Environment.REGISTRY_FACTORY_CLASS, 		getRegistryFactoryClass()),
			new KeyValuePair(Environment.REGISTRY_USER, 				getRegistryUser()),
			new KeyValuePair(Environment.REGISTRY_PASSWORD, 			getRegistryPassword()),
			new KeyValuePair(Environment.REGISTRY_SCOUT_TRANSPORT_CLASS,getRegistryScoutTransportClass()),
			new KeyValuePair(Environment.PARAMS_REPOS_IMPL_CLASS,		getParamRepositoryImplClass()),
			new KeyValuePair(Environment.OBJECT_STORE_CONFIG_FILE, 		getObjStoreConfigFile()),
			new KeyValuePair(Environment.ENCRYPT_FACTORY_CLASS, 		getEncryptionFactoryClass()),
            new KeyValuePair(Environment.LOAD_BALANCER_POLICY,          getLoadBalancerPolicy()),
            new KeyValuePair(Environment.REDELIVER_DLS_SERVICE_ON,      getRedeliveryDlsOn()),
            new KeyValuePair(Environment.REGISTRY_CACHE_LIFE_MILLIES,   getRegistryCacheLife()),
			new KeyValuePair(Environment.MSG_STORE_DB_CONNECTION_URL, 	getStoreUrl()),
			new KeyValuePair(Environment.MSG_STORE_DB_JDBC_DRIVER, 		getStoreDriver()),
			new KeyValuePair(Environment.MSG_STORE_DB_CONNECTION_USER, 	getStoreUser()),
			new KeyValuePair(Environment.MSG_STORE_DB_CONNECTION_PWD, 	getStorePwd()),
			new KeyValuePair(Environment.MSG_STORE_DB_POOL_MIN_SIZE, 	getStorePoolMinSize()),
			new KeyValuePair(Environment.MSG_STORE_DB_POOL_MAX_SIZE, 	getStorePoolMaxSize()),
			new KeyValuePair(Environment.MSG_STORE_DB_POOL_INITIAL_SIZE,getStorePoolInitialSize()),
			new KeyValuePair(Environment.MSG_STORE_DB_POOL_TEST_TABLE,  getStorePoolTestTable()),
			new KeyValuePair(Environment.MSG_STORE_DB_POOL_TIMEOUT_MILLIS,getStorePoolTimeoutMillis()),
			new KeyValuePair(Environment.MSG_STORE_DB_CONN_MANAGER		,getStoreDBConnectionManager()),
			new KeyValuePair(Environment.MSG_STORE_DB_DATASOURCE_NAME	,getStoreDBDatasourceName())
	};

	public static String dump()
	{
		StringBuilder sb = new StringBuilder("Dump of SystemProperties:\n");
		for (KeyValuePair oCurr : s_oaKV)
		{
			sb.append(oCurr.getKey()).append("=").append(oCurr.getValue())
					.append("\n");
		}
		return sb.append("______________________________________").toString();
	} // ________________________________

	public static String getSmtpHost()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.SMTP_HOST, Environment.DEFAULT_HOST);
	}

	public static String getSmtpUsername()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.SMTP_USERNAME,
				Environment.DEFAULT_USERNAME);
	}

	public static String getSmtpPassword()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.SMTP_PASSWORD,
				Environment.DEFAULT_PASSWORD);
	}

	public static String getSmtpPort()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.SMTP_PORT, Environment.DEFAULT_PORT);
	}

	public static String getSmtpFrom()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.SMTP_FROM);
	}

	public static String getSmtpAuth()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.SMTP_AUTH);
	}

	public static String getJndiServerURL()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.JNDI_SERVER_URL,
				Environment.DEFAULT_HOST);
	}
	
	public static String getJndiServerContextFactory()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.JNDI_SERVER_CONTEXT_FACTORY,
				Environment.DEFAULT_JNDI_CONTEXT_FACTORY);
	}
	
	public static String getJndiServerPkgPrefix()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.JNDI_SERVER_PKG_PREFIX,
				Environment.DEFAULT_JNDI_PKG_PREFIX);
	}
    
    public static String getLoadBalancerPolicy()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.LOAD_BALANCER_POLICY,
                Environment.DEFAULT_LOAD_BALANCER_POLICY);
    }
    
    public static String getRedeliveryDlsOn()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.REDELIVER_DLS_SERVICE_ON,
                    Environment.DEFAULT_REDELIVER_DLS_ON);
    }
    
    public static String getRegistryCacheLife()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.REGISTRY_CACHE_LIFE_MILLIES,
                    Environment.DEFAULT_REGISTRY_CACHE_LIFE_MILLIES);
    }
	/**
	 * The Registry Query Manager URI can be used to obtain information about services and their endPoints.
	 * @return Registry Query Manager URI String
	 */
	public static String getRegistryQueryManageURI()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_QUERY_MANAGER_URI);
	}
	/**
	 * The Registry Lifecycle Manager URI can be use to publish services.
	 * @return Registry Lifecycle Manager URI String
	 */
	public static String getRegistryLifecycleManagerURI()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_LIFECYCLE_MANAGER_URI);
	}
	/**
	 * The Registry Factory Class setting specifies which JAXR implementation should be used.
	 * @return Registry Factory Class String
	 */
	public static String getRegistryFactoryClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_FACTORY_CLASS);
	}
	/**
	 * The Registry Implememtation Class defines wrapper class
	 * @return Registry Implementation Class String
	 */
	public static String getRegistryImplementationClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_IMPEMENTATION_CLASS);
	}
	/**
	 * The Registry Scout Transport Class defines which communication protocol Scout should use to communicate
	 * with the UDDI registry. Note that this parameter is Scout specific.
	 * @return Registry Scout Transport Class String
	 */
	public static String getRegistryScoutTransportClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_SCOUT_TRANSPORT_CLASS);
	}
	/**
	 * The Registry User is used to inquire and publish to the Registry.
	 * @return Registry User String
	 */
	public static String getRegistryUser()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_USER);
	}
	/**
	 * The Registry Password that goes with the Registry User.
	 * @return Registry Password String
	 */
	public static String getRegistryPassword()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_PASSWORD);
	}
    
    public static String getRegistrySemanticEquivalences()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_SEMANTIC_EQUIVALENCES);
    }
    
    public static String getRegistryPostalAddressScheme()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_POSTAL_ADDRESS_SCHEME);
    }
    
    public static String getRegistrySecurityAuthenticationMethod()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_SECURITY_AUTHENTICATION_METHOD);
    }
    
    public static String getRegistryUDDIMaxRows()
    {
        return ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_UDDI_MAX_ROWS);
    }
	
	public static String getParamRepositoryImplClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.PARAMS_REPOS_IMPL_CLASS);
	}

	public static String getObjStoreConfigFile()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.OBJECT_STORE_CONFIG_FILE);
		return property;
	}

	public static String getEncryptionFactoryClass()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperty(Environment.ENCRYPT_FACTORY_CLASS,
				org.jboss.soa.esb.services.DefaultEncryptionFactory.class
						.getName());
	}
	
	public static String getStoreUrl()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_CONNECTION_URL);
		return property;
	}
	
	public static String getStoreDriver()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_JDBC_DRIVER);
		return property;
	}	
	
	public static String getStorePwd()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_CONNECTION_PWD);
		return property;
	}
	
	public static String getStoreUser()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_CONNECTION_USER);
		return property;
	}
	
	public static String getStorePoolInitialSize()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_POOL_INITIAL_SIZE);
		return property;
	}
	
	public static String getStorePoolMinSize()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_POOL_MIN_SIZE);
		return property;
	}
	
	public static String getStorePoolMaxSize()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_POOL_MAX_SIZE);
		return property;
	}
	
	public static String getStorePoolTestTable()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_POOL_TEST_TABLE);
		return property;
	}
	
	public static String getStorePoolTimeoutMillis()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_POOL_TIMEOUT_MILLIS);
		return property;
	}
	
	public static String getStoreDBConnectionManager()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_CONN_MANAGER);
		return property;
	}
	
	public static String getStoreDBDatasourceName()
	{
		String property = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_DB_DATASOURCE_NAME);
		return property;
	}
	
	public static String getJcrStoreJNDIPath()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_JCR_JNDI_PATH);
	}
	
	public static String getJcrStoreUsername()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_JCR_USERNAME);
	}
	
	public static String getJcrStorePassword()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_JCR_PASSWORD);
	}
	
	public static String getJcrStoreRootNodePath()
	{
		return ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE).getProperty(Environment.MSG_STORE_JCR_ROOT_NODE_PATH);
	}
	
	/**
	 * Construct a naming context based on the connection details outlined
	 * in the named configuration module/section.
	 * <p/>
	 * The module/section parameters are:
	 * <ol>
	 * 	<li><b>{@link Context#PROVIDER_URL}</b>: Value defaults to "{@link Environment#JBOSS_PROVIDER_URL}".</li>
	 * 	<li><b>{@link Context#INITIAL_CONTEXT_FACTORY}</b>: Value defaults to "{@link Environment#JBOSS_INITIAL_CONTEXT_FACTORY}".</li>
	 * 	<li><b>{@link Context#URL_PKG_PREFIXES}</b>: Value defaults to "{@link Environment#JBOSS_URL_PKG_PREFIX}".</li>
	 * </ol>
	 * 
	 * @param contextModuleName Conficuration module/section name.
	 * @return The context instance.
	 * @throws ConfigurationException Unable to connect to context.
	 * 
	 * @deprecated {@link org.jboss.soa.esb.helpers.NamingContextPool}
	 */
	public static Context getNamingContext(String contextModuleName) throws ConfigurationException {
	    final Properties environment = getProperties(contextModuleName) ;
		
        try {
            return new InitialContext(environment) ;
        } catch (final NamingException ne) {
            throw new ConfigurationException("Failed to create JNDI context [" + contextModuleName + "].");
        }
	}
	
	private static Properties getProperties(final String contextModuleName)
	{
            final String providerUrl = ModulePropertyManager.getPropertyManager(contextModuleName).getProperty(Context.PROVIDER_URL, Environment.JBOSS_PROVIDER_URL);
            final String initialContextFactory = ModulePropertyManager.getPropertyManager(contextModuleName).getProperty(Context.INITIAL_CONTEXT_FACTORY, Environment.JBOSS_INITIAL_CONTEXT_FACTORY);
            final String urlPackagePrefix = ModulePropertyManager.getPropertyManager(contextModuleName).getProperty(Context.URL_PKG_PREFIXES, Environment.JBOSS_URL_PKG_PREFIX);
            final Properties environment = new Properties();
            environment.setProperty(Context.PROVIDER_URL, providerUrl);
            environment.setProperty(Context.INITIAL_CONTEXT_FACTORY, initialContextFactory);
            environment.setProperty(Context.URL_PKG_PREFIXES, urlPackagePrefix);
            return environment ;
	}
	
	/**
	 * Lookup the JMS Connection Factory based on the connection details outlined
	 * in the named configuration module/section.
	 * <p/>
	 * The module/section parameters are:
	 * <ol>
	 * 	<li><b>{@link Context#PROVIDER_URL}</b>: Value defaults to "{@link Environment#JBOSS_PROVIDER_URL}".</li>
	 * 	<li><b>{@link Context#INITIAL_CONTEXT_FACTORY}</b>: Value defaults to "{@link Environment#JBOSS_INITIAL_CONTEXT_FACTORY}".</li>
	 * 	<li><b>{@link Context#URL_PKG_PREFIXES}</b>: Value defaults to "{@link Environment#JBOSS_URL_PKG_PREFIX}".</li>
	 * 	<li><b>javax.jms.ConnectionFactory</b>: Value defaults to "".</li>
	 * </ol>
	 * 
	 * @param jmsConnectionFactoryModuleName Configuration module/section name.
	 * @return The JMS ConnectionFactory instance.
	 * @throws ConfigurationException Lookup failed either because it was unable to 
	 * lookup the context, or the ConnectionFactory lookup failed.
	 */
	public static ConnectionFactory getJmsConnectionFactory(String jmsConnectionFactoryModuleName) throws ConfigurationException {
	        final Properties environment = getProperties(jmsConnectionFactoryModuleName) ;
	        try {
	            Context context = NamingContextPool.getNamingContext(environment);
	            try {
                        String connectionFactoryRuntime = ModulePropertyManager.getPropertyManager(jmsConnectionFactoryModuleName).getProperty(ConnectionFactory.class.getName(), "ConnectionFactory");
                        ConnectionFactory factory = null;
                        
                        try {
                                factory = (ConnectionFactory) context.lookup(connectionFactoryRuntime);
                        } catch (NamingException e) {
                                throw new ConfigurationException("JNDI lookup of JMS Connection Factory [" + connectionFactoryRuntime + "] failed.", e);
                        } catch (ClassCastException e) {
                                throw new ConfigurationException("JNDI lookup of JMS Connection Factory failed.  Class [" + connectionFactoryRuntime + "] is not an instance of [" + ConnectionFactory.class.getName() + "].", e);
                        }
                        return factory;
	            } finally {
	                NamingContextPool.releaseNamingContext(context) ;
	            }
	        } catch (final NamingContextException nce) {
	            throw new ConfigurationException("Unexpected exception while accessing naming context pool", nce) ;
	        }
        }
}
