/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.addressing.eprs;

import java.net.URI;
import java.net.URISyntaxException;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.PortReference;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * This class is a helper class for using Hibernate EPRs.
 *
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 */
public class HibernateEpr extends EPR {
	// TODO: Need to remove String constants and getter/setters for messagefield,
	// status-field, data-field, insert-timestamp-field, postDelete, errorDelete 
	// all of which are unused.
	
	public static final String LOAD_LISTENER = "loadListener";
	public static final String SAVE_LISTENER = "saveListener";
	public static final String DELETE_LISTENER = "deleteListener";

	public static final String CLASS_NAME_TAG 			= "classname";
	public static final String EVENT_TAG				= "event";
	public static final String HIBERNATE_CFG_TAG		= "hibernate-cfg-file";
		
	
	
	public static final String MESSAGE_FIELD_TAG 		= "message-field";
	public static final String STATUS_FIELD_TAG 		= "status-field";
	public static final String DATA_FIELD_TAG 			= "data-field";
	public static final String TIMESTAMP_FIELD_TAG 		= "insert-timestamp-field";
	public static final String POST_DEL_TAG				= "postDelete";
	public static final String ERROR_DEL_TAG			= "errorDelete";
	

	public static final String DEFAULT_REPLY_TO_CLASS_SUFFIX = "_reply_table";
	
	private boolean loadListener = true;
	private boolean saveListener = true;
	private boolean deleteListener = true;
	private boolean cfgFileSet = false;
	private boolean classNameSet = false;
	private boolean statusSet = false;
	private boolean messageSet = false;
	private boolean timestampSet = false;
	private boolean dataSet = false;

	private static URI m_type;

	
	public HibernateEpr(EPR epr) {
		super(epr);
		
		if (epr instanceof HibernateEpr)
		{
		    HibernateEpr instance = (HibernateEpr) epr;
		    
		    cfgFileSet = instance.cfgFileSet;
		    classNameSet = instance.classNameSet;
		    statusSet = instance.statusSet;
		    messageSet = instance.messageSet;
		    timestampSet = instance.timestampSet;
		    dataSet = instance.dataSet;
		}
	}
	
	public HibernateEpr (EPR epr, Element header)
	{
		super(epr);
		
		NodeList nl = header.getChildNodes();
		for (int i = 0; i < nl.getLength(); i++)
		{
			try
			{
				String tag = nl.item(i).getLocalName();
				
				if (tag != null)
				{
					if (tag.equals(HIBERNATE_CFG_TAG)) {
					    setHibernateCfgFile(nl.item(i).getTextContent());
					} else if (tag.equals(CLASS_NAME_TAG)) {
					    setClassName(nl.item(i).getTextContent());
					} else if (tag.equals(EVENT_TAG)) {
					    setEvent(nl.item(i).getTextContent());
					}
				}
			} catch (Exception ex) {
				ex.printStackTrace();
			}
		}
	}
	
	@Override
	public PortReference getAddr() {
		// TODO Auto-generated method stub
		return super.getAddr();
	}

	@Override
	public void setAddr(PortReference uri) {
		// TODO Auto-generated method stub
		super.setAddr(uri);
	}
	
	/**
	 * @return the driver used by this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getHibernateCfgFile () throws URISyntaxException
	{
		return getAddr().getExtensionValue(HIBERNATE_CFG_TAG);
	}
	
	/**
	 * Set the tablename that is used by this EPR.
	 * 
	 * @param cfgFile the table name.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setHibernateCfgFile(String cfgFile) throws URISyntaxException
	{
		if (cfgFile == null)
			throw new IllegalArgumentException();
		
		if (cfgFileSet)
			throw new IllegalStateException("Table name already set.");
		
		getAddr().addExtension(HIBERNATE_CFG_TAG, cfgFile);
		cfgFileSet = true;
	}
	
	/**
	 * Set the tablename that is used by this EPR.
	 * 
	 * @param className the table name.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setClassName (String className) throws URISyntaxException
	{
		if (className == null)
			throw new IllegalArgumentException();
		
		if (classNameSet)
			throw new IllegalStateException("Table name already set.");
		
		getAddr().addExtension(CLASS_NAME_TAG, className);
		classNameSet = true;
	}
	
	/**
	 * @return the table name used by this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getClassName () throws URISyntaxException
	{
		return getAddr().getExtensionValue(CLASS_NAME_TAG);
	}
	
	
	
	
	/**
	 * Set the event that is used by this EPR.
	 * 
	 * @param eventName the table name.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setEvent (String eventName) throws URISyntaxException
	{
		if (eventName == null)
			throw new IllegalArgumentException();
		
		if (classNameSet)
			throw new IllegalStateException("Table name already set.");
		
		getAddr().addExtension(EVENT_TAG, eventName);
		classNameSet = true;
	}
	
	/**
	 * @return the table name used by this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getEvent () throws URISyntaxException
	{
		return getAddr().getExtensionValue(EVENT_TAG);
	}
	
	/**
	 * Set the status column that is used by this EPR.
	 * 
	 * @param statusField the status field.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setStatusField (String statusField) throws URISyntaxException
	{
		if (statusField == null)
			throw new IllegalArgumentException();
		
		if (statusSet)
			throw new IllegalStateException("Status column already set.");
		
		getAddr().addExtension(STATUS_FIELD_TAG, statusField);
		statusSet = true;
	}
	
	/**
	 * @return the status column name used by this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getStatusField () throws URISyntaxException
	{
		return getAddr().getExtensionValue(STATUS_FIELD_TAG);
	}
	
	/**
	 * 
	 * @param messageFieldName 
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setMessageField (String messageFieldName) throws URISyntaxException
	{
		if (messageFieldName == null)
			throw new IllegalArgumentException();
		
		if (messageSet)
			throw new IllegalStateException("Message Id column already set.");
		
		getAddr().addExtension(MESSAGE_FIELD_TAG, messageFieldName);
		messageSet = true;
	}
	
	/**
	 * @return the message id column used by this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getMessageField () throws URISyntaxException
	{
		return getAddr().getExtensionValue(MESSAGE_FIELD_TAG);
	}
	
	
	/**
	 * Set the column that is used by this EPR to store timestamp when the message was inserted.
	 * 
	 * @param timeField the column name.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setTimestampField (String timeField) throws URISyntaxException
	{
		if (timeField == null)
			throw new IllegalArgumentException();
		
		if (timestampSet)
			throw new IllegalStateException("Timestamp column already set.");
		
		getAddr().addExtension(TIMESTAMP_FIELD_TAG, timeField);
		timestampSet = true;
	}
	
	/**
	 * @return the timestamp column name used by this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getTimestampColumn () throws URISyntaxException
	{
		return getAddr().getExtensionValue(TIMESTAMP_FIELD_TAG);
	}
	
	/**
	 * Set the field that is used by this EPR to store message data.
	 * 
	 * @param dataField the column name.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setDataField (String dataField) throws URISyntaxException
	{
		if (dataField == null)
			throw new IllegalArgumentException();
		
		if (dataSet)
			throw new IllegalStateException("Data field already set.");
		
		getAddr().addExtension(DATA_FIELD_TAG, dataField);
		dataSet = true;
	}
	
	/**
	 * @return the data column name used by this EPR.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final String getDataField () throws URISyntaxException
	{
		return getAddr().getExtensionValue(DATA_FIELD_TAG);
	}
	
	
	public boolean isDeleteListener() {
		return deleteListener;
	}

	public void setDeleteListener(boolean deleteListener) {
		this.deleteListener = deleteListener;
	}

	public boolean isLoadListener() {
		return loadListener;
	}

	public void setLoadListener(boolean loadListener) {
		this.loadListener = loadListener;
	}

	public boolean isSaveListener() {
		return saveListener;
	}

	public void setSaveListener(boolean saveListener) {
		this.saveListener = saveListener;
	}	
	
	public String toString ()
	{
		return "HibernateEpr [ "+super.getAddr().extendedToString()+" ]";
	}

	public EPR copy ()
	{
	    return new HibernateEpr(this);
	}
	
	protected final void changeClassName (String className) throws URISyntaxException
	{
	    classNameSet = false;
	    
	    super.getAddr().removeExtension(CLASS_NAME_TAG, getClassName());
	    
	    setClassName(className);
	}
	
	public static final URI type ()
	{
	    return m_type;
	}
	
	static
	{
	    try
		{
		    m_type = new URI("urn:jboss/esb/epr/type/hibernate");
		}
		catch (Exception ex)
		{
		    ex.printStackTrace();
		    
		    throw new ExceptionInInitializerError(ex.toString());
		}
	}
}