/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.naming.strategy;

import org.jboss.soa.esb.message.Message;
import org.w3c.dom.Node;

/**
 * Naming strategy that changes the suffix of a file to a configured value.
 * 
 * @author Derek Adams
 */
public class ChangeSuffixNamingStrategy extends AbstractFileNamingStrategy {

	/** New suffix to use */
	private String newSuffix = DEFAULT_NEW_SUFFIX;

	/** Default suffix if none is configured */
	public static final String DEFAULT_NEW_SUFFIX = "esbout";

	/** XML attribute that holds the new suffix */
	public static final String ATTR_NEW_SUFFIX = "newSuffix";

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.actions.naming.strategy.FileNamingStrategy#configure(org.w3c.dom.Node)
	 */
	public void configure(Node xml) {
		Node newSuffixNode = xml.getAttributes().getNamedItem(ATTR_NEW_SUFFIX);
		if (newSuffixNode != null) {
			setNewSuffix(newSuffixNode.getNodeValue());
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.actions.naming.strategy.AbstractFileNamingStrategy#process(java.lang.String,
	 *      org.jboss.soa.esb.message.Message)
	 */
	public String process(String filename, Message message) {
		return getFilenameWithoutSuffix(filename) + '.' + getNewSuffix();
	}

	/**
	 * Set the new suffix to use.
	 * 
	 * @param newSuffix
	 */
	public void setNewSuffix(String newSuffix) {
		this.newSuffix = newSuffix;
	}

	/**
	 * Get the new suffix to use.
	 * 
	 * @return String
	 */
	public String getNewSuffix() {
		return newSuffix;
	}
}