/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. 
 * See the copyright.txt in the distribution for a full listing 
 * of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU General Public License, v. 2.0.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License,
 * v. 2.0 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */
package org.jboss.internal.soa.esb.util;

import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import org.jboss.internal.soa.esb.util.stax.ParsingSupport;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.util.StringPropertyReplacer;

/**
 * Helper class for manipulating XML documents.
 * 
 * @author <a href='mailto:kevin.conner@jboss.com'>Kevin Conner</a>
 */
public class XMLHelper
{
    /**
     * The XML input factory.
     */
    private static final XMLInputFactory XML_INPUT_FACTORY = getXMLInputFactory() ;
    /**
     * The XML output factory.
     */
    private static final XMLOutputFactory XML_OUTPUT_FACTORY = XMLOutputFactory.newInstance() ;

    /**
     * Get the XML stream reader.
     * @param reader The input reader.
     * @return The XML stream reader.
     * @throws XMLStreamException For errors obtaining an XML stream reader.
     */
    public static XMLStreamReader getXMLStreamReader(final Reader reader)
        throws XMLStreamException
    {
        return XML_INPUT_FACTORY.createXMLStreamReader(reader) ;
    }

    /**
     * Get the XML stream reader.
     * @param is The input stream.
     * @return The XML stream reader.
     * @throws XMLStreamException For errors obtaining an XML stream reader.
     */
    public static XMLStreamReader getXMLStreamReader(final InputStream is)
        throws XMLStreamException
    {
        return XML_INPUT_FACTORY.createXMLStreamReader(is) ;
    }

    /**
     * Get the XML stream reader.
     * @param is The input stream.
     * @param encoding The input stream encoding.
     * @return The XML stream reader.
     * @throws XMLStreamException For errors obtaining an XML stream reader.
     */
    public static XMLStreamReader getXMLStreamReader(final InputStream is, final String encoding)
        throws XMLStreamException
    {
        return XML_INPUT_FACTORY.createXMLStreamReader(is, encoding) ;
    }

    /**
     * Get the XML stream writer.
     * @param writer The output writer.
     * @return The XML stream writer.
     * @throws XMLStreamException For errors obtaining an XML stream writer.
     */
    public static XMLStreamWriter getXMLStreamWriter(final Writer writer)
        throws XMLStreamException
    {
        return XML_OUTPUT_FACTORY.createXMLStreamWriter(writer) ;
    }

    /**
     * Get the XML stream writer.
     * @param os The output stream.
     * @return The XML stream writer.
     * @throws XMLStreamException For errors obtaining an XML stream writer.
     */
    public static XMLStreamWriter getXMLStreamWriter(final OutputStream os)
        throws XMLStreamException
    {
        return XML_OUTPUT_FACTORY.createXMLStreamWriter(os) ;
    }

    /**
     * Get the XML stream writer.
     * @param os The output stream.
     * @param encoding The output stream encoding.
     * @return The XML stream writer.
     * @throws XMLStreamException For errors obtaining an XML stream writer.
     */
    public static XMLStreamWriter getXMLStreamWriter(final OutputStream os, final String encoding)
        throws XMLStreamException
    {
        return XML_OUTPUT_FACTORY.createXMLStreamWriter(os, encoding) ;
    }
    
    /**
     * Replace system property values within the attribute values/text elements.
     * @param streamReader The XML stream reader.
     * @param streamWriter The XMl stream writer.
     * @throws XMLStreamException For errors during parsing.
     */
    public static void replaceSystemProperties(final XMLStreamReader streamReader,
        final XMLStreamWriter streamWriter)
        throws XMLStreamException
    {
        streamWriter.writeStartDocument() ;
        
        StreamHelper.skipToStartElement(streamReader) ;
        final QName elementName = streamReader.getName() ;
        final String uri = StreamHelper.writeStartElement(streamWriter, elementName) ;
        
        new SystemPropertyReplacementParser(streamReader, streamWriter) ;
        
        StreamHelper.writeEndElement(streamWriter, elementName.getPrefix(), uri) ;
        
        streamWriter.writeEndDocument() ;
        streamWriter.flush() ;
    }

    /**
     * Create the XML input factory.
     * @return The XML input factory.
     */
    private static XMLInputFactory getXMLInputFactory()
    {
        final XMLInputFactory xmlInputFactory = XMLInputFactory.newInstance() ;
        xmlInputFactory.setProperty(XMLInputFactory.IS_COALESCING, Boolean.TRUE) ;
        return xmlInputFactory ;
    }
    
    /**
     * The parser class used to perform system property replacement.
     * @author kevin
     */
    private static final class SystemPropertyReplacementParser extends ParsingSupport
    {
        /**
         * The output writer.
         */
        private final XMLStreamWriter out ;
        
        /**
         * Construct the parser.
         * @param in The XML input stream.
         * @param out The XML output stream.
         * @throws XMLStreamException For errors during parsing.
         */
        SystemPropertyReplacementParser(final XMLStreamReader in, final XMLStreamWriter out)
            throws XMLStreamException
        {
            this.out = out ;
            parse(in) ;
        }
        
        /**
         * Set the text value of this element.
         * @param in The current input stream.
         * @param value The text value of this element.
         * @throws XMLStreamException For errors during parsing.
         */
        protected void putValue(final XMLStreamReader in, final String value)
            throws XMLStreamException
        {
            out.writeCharacters(StringPropertyReplacer.replaceProperties(value)) ;
        }
        
        /**
         * Add the attribute value.
         * @param in The current input stream.
         * @param attributeName The qualified attribute name.
         * @param attributeValue The qualified attribute value.
         * @throws XMLStreamException For errors during parsing.
         */
        protected void putAttribute(final XMLStreamReader in,
            final QName attributeName, final String attributeValue)
            throws XMLStreamException
        {
            StreamHelper.writeAttribute(out, attributeName, StringPropertyReplacer.replaceProperties(attributeValue)) ;
        }
        
        /**
         * Add the element.
         * @param in The current input stream.
         * @param elementName The qualified element name.
         * @throws XMLStreamException For errors during parsing.
         */
        protected void putElement(final XMLStreamReader in,
            final QName elementName)
            throws XMLStreamException
        {
            final String uri = StreamHelper.writeStartElement(out, elementName) ;
            new SystemPropertyReplacementParser(in, out) ;
            StreamHelper.writeEndElement(out, elementName.getPrefix(), uri) ;
        }
    }
}
