/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.internal.soa.esb.services.registry;

import java.io.UnsupportedEncodingException;
import java.net.PasswordAuthentication;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import javax.xml.registry.BulkResponse;
import javax.xml.registry.BusinessLifeCycleManager;
import javax.xml.registry.BusinessQueryManager;
import javax.xml.registry.Connection;
import javax.xml.registry.ConnectionFactory;
import javax.xml.registry.FindQualifier;
import javax.xml.registry.JAXRException;
import javax.xml.registry.JAXRResponse;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.EmailAddress;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.Organization;
import javax.xml.registry.infomodel.PersonName;
import javax.xml.registry.infomodel.PostalAddress;
import javax.xml.registry.infomodel.RegistryObject;
import javax.xml.registry.infomodel.Service;
import javax.xml.registry.infomodel.ServiceBinding;
import javax.xml.registry.infomodel.TelephoneNumber;
import javax.xml.registry.infomodel.User;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.addressing.helpers.EPRHelper;
import org.jboss.soa.esb.MarshalException;
import org.jboss.soa.esb.UnmarshalException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.services.registry.Registry;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;
/**
 * Utility class for the Registry.
 * If need be we can extract the interface from here, add a factory and have JAXR as a plugin, allowing
 * for other RegistryAPIs.
 *
 * @author Kurt Stam
 */
public class JAXRRegistryImpl implements Registry
{
	private static Logger logger = Logger.getLogger(JAXRRegistryImpl.class);
	public static Set<PasswordAuthentication> creds = new HashSet<PasswordAuthentication>();
	private static Properties props = new Properties();
	private static Organization jbossESBOrganization;
	
	private synchronized static void init() 
	{
	    props = new Properties();
	    props.setProperty("javax.xml.registry.queryManagerURL", Configuration.getRegistryQueryManageURI());
	    props.setProperty("javax.xml.registry.lifeCycleManagerURL", Configuration.getRegistryLifecycleManagerURI());
	    props.setProperty("javax.xml.registry.factoryClass", Configuration.getRegistryFactoryClass());
        if (Configuration.getRegistrySemanticEquivalences()!=null) {
            props.setProperty("javax.xml.registry.semanticEquivalences", Configuration.getRegistrySemanticEquivalences());
        }
        if (Configuration.getRegistryPostalAddressScheme()!=null) {
            props.setProperty("javax.xml.registry.postalAddressScheme", Configuration.getRegistryPostalAddressScheme());
        }
        if (Configuration.getRegistrySecurityAuthenticationMethod()!=null) {
            props.setProperty("javax.xml.registry.security.authenticationMethod", Configuration.getRegistrySecurityAuthenticationMethod());
        }
        if (Configuration.getRegistryUDDIMaxRows()!=null) {
            props.setProperty("javax.xml.registry.uddi.maxRows", Configuration.getRegistryUDDIMaxRows());
        }
        if (Configuration.getRegistryScoutTransportClass()!=null) {
            props.setProperty("scout.proxy.transportClass", Configuration.getRegistryScoutTransportClass());
        }
	    String user = Configuration.getRegistryUser();
	    String password = Configuration.getRegistryPassword();
	    PasswordAuthentication passwdAuth = new PasswordAuthentication(user, password.toCharArray());
        creds.add(passwdAuth);
	}

	/** 
	 * Creates a connecton to a JAXR capable registy.
	 * 
	 * @return Connection to a Registry using JAXR. 
	 */
	protected synchronized static Connection getConnection() 
	{
		Connection connection = null;
		init();
	    try
	    {   // Create the connection, passing it the configuration properties
	        ConnectionFactory factory = ConnectionFactory.newInstance();
	        factory.setProperties(props);
	        connection = factory.createConnection();
	    } catch (JAXRException e) {
            logger.log(Level.ERROR, "Could not set up a connection to the Registry. " + e.getMessage(), e);
	    }
	    return connection;
	}
	/**
	 * Closes the connection to the Registry
	 */
	protected synchronized static void closeConnection(Connection connection)
	{
		try {
			if (connection!=null && !connection.isClosed()) {
				connection.close();
			}
		} catch (JAXRException je) {
			logger.log(Level.ERROR, je.getMessage(), je);
		}
	}
	/** 
	 * Publish an EPR to the Registry
	 */
	protected Service registerService(String category, String serviceName, String serviceDescription) throws JAXRException 
	{
		Service service =null;
		Organization organization = getJBossESBOrganization();
		Connection connection = JAXRRegistryImpl.getConnection();
		try {
			connection.setCredentials(JAXRRegistryImpl.creds);
			RegistryService rs = connection.getRegistryService();
            BusinessQueryManager bqm = rs.getBusinessQueryManager();
			BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
		    //Adding the category as prefix for the name
			service = blm.createService(blm.createInternationalString(serviceName));
			service.setDescription(blm.createInternationalString(serviceDescription));
            Collection<String> findQualifiers = new ArrayList<String>();
            findQualifiers.add(FindQualifier.AND_ALL_KEYS);
            findQualifiers.add(FindQualifier.SORT_BY_NAME_DESC);
			ClassificationScheme cScheme = bqm.findClassificationSchemeByName(findQualifiers, "org.jboss.soa.esb.:category");
            Classification classification = blm.createClassification(cScheme, "category", category);
			service.addClassification(classification);
			organization.addService(service);
			saveRegistryObject(service);
		} finally {
			closeConnection(connection);
		}
		return service;
	}
	/** 
	 * Remove an EPR from the Registry
	 */
	@SuppressWarnings("unchecked")
    public void unRegisterService(String category, String serviceName) throws RegistryException, ServiceNotFoundException{
//    first find the ServiceBindings for this service
        Connection connection = JAXRRegistryImpl.getConnection();
        Service service = null;
        try {
            service = findService(category, serviceName);
            if (service==null) {
                throw new ServiceNotFoundException("No such EPR found for service with name = " 
                        + serviceName);
            }
            connection.setCredentials(JAXRRegistryImpl.creds);
            RegistryService rs = connection.getRegistryService();
            BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
            Collection<Key> serviceKeys = new ArrayList<Key>();
            serviceKeys.add(service.getKey());
            blm.deleteServices(serviceKeys);
        } catch (JAXRException je) {
            throw new RegistryException(je.getLocalizedMessage(), je);
        } finally {
            closeConnection(connection);
        }
	}
	/** 
	 * Publish an EPR to the Registry
	 */
	public void registerEPR(String category, String serviceName, String serviceDescription, EPR epr, String eprDescription) 
		throws RegistryException
	{
		Connection connection = JAXRRegistryImpl.getConnection();
		try {
			//Find the service
			Service service = findService(category,serviceName);
			if (service==null) {
				logger.log(Level.DEBUG, "Service " + serviceName + " does not yet exist, creating now..");
				service = registerService(category, serviceName, serviceDescription);
			}

			connection.setCredentials(JAXRRegistryImpl.creds);
			RegistryService rs = connection.getRegistryService();
			BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
            BusinessQueryManager bqm = rs.getBusinessQueryManager();
			ServiceBinding serviceBinding = blm.createServiceBinding();
			serviceBinding.setDescription(blm.createInternationalString(eprDescription));
			String xml = EPRHelper.toXMLString(epr);
			serviceBinding.setAccessURI(xml);
		
		    ArrayList<ServiceBinding> serviceBindings = new ArrayList<ServiceBinding>();
		    serviceBindings.add(serviceBinding);
		    service.addServiceBindings(serviceBindings);
            Collection<String> findQualifiers = new ArrayList<String>();
            findQualifiers.add(FindQualifier.AND_ALL_KEYS);
            findQualifiers.add(FindQualifier.SORT_BY_NAME_DESC);
            ClassificationScheme cScheme = bqm.findClassificationSchemeByName(findQualifiers, "org.jboss.soa.esb.:category");
            Classification classification = blm.createClassification(cScheme, "category", category);
            service.addClassification(classification);
           
			saveRegistryObject(serviceBinding);
		} catch (JAXRException je) {
			throw new RegistryException(je.getLocalizedMessage(), je);
        } catch (MarshalException me) {
            throw new RegistryException(me.getLocalizedMessage(), me);
		} finally {
			closeConnection(connection);
		}
	}
	/** 
	 * Remove an EPR from the Registry
	 */
	public void unRegisterEPR(String category, String serviceName, EPR toBeDeletedEPR) throws RegistryException, ServiceNotFoundException{
		//first find the ServiceBindings for this service
        Connection connection = JAXRRegistryImpl.getConnection();
        Service service = null;
		try {
            service = findService(category, serviceName);
            if (service==null) {
                throw new ServiceNotFoundException("No such Service found for service with category= "
                        + category + " and name = " + serviceName);
            }
            connection.setCredentials(JAXRRegistryImpl.creds);
			Collection serviceBindings = findServiceBindings(service);
			service.addServiceBindings(serviceBindings);
			for (Iterator i=serviceBindings.iterator();i.hasNext();){
				ServiceBinding serviceBinding = (ServiceBinding) i.next();
				String eprXML = serviceBinding.getAccessURI().trim();
                // for backwards compatibility still have the decoder if
                // unmarchalling fails
                try {
                    EPR epr = EPRHelper.fromXMLString(eprXML);
                } catch (UnmarshalException unme) {
                    try {
                        eprXML = URLDecoder.decode(eprXML, "UTF-8").trim();
                    } catch (UnsupportedEncodingException ue) {
                        logger.error(unme.getMessage(), unme);
                        logger.error(ue.getMessage(), ue);
                    }
                }
                String toBeDeletedEPRXml = EPRHelper.toXMLString(toBeDeletedEPR).trim();
				if (eprXML.equals(toBeDeletedEPRXml)) {
                    RegistryService rs = connection.getRegistryService();
                    BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
                    Collection<Key> serviceBindingKeys = new ArrayList<Key>();
                    serviceBindingKeys.add(serviceBinding.getKey());
                    blm.deleteServiceBindings(serviceBindingKeys);
					return;
				}
			}
            throw new RegistryException("No such EPR found for service with name = " 
                    + serviceName + " and EPR=" + toBeDeletedEPR);
		} catch (JAXRException je) {
			throw new RegistryException(je.getLocalizedMessage(), je);
        } catch (MarshalException me) {
            throw new RegistryException(me.getLocalizedMessage(), me);
		} finally {
		    closeConnection(connection);
        }
	}
	
	/** 
	 * {@inheritDoc}
	 * @return collection services
	 */
	public List<String> findAllServices() throws RegistryException
	{
		List<String> serviceNames = new ArrayList<String>();
		try {
			Collection services = getJBossESBOrganization().getServices();
			for (Iterator i=services.iterator();i.hasNext();) {
				String serviceName = ((Service)i.next()).getName().getValue();
				serviceNames.add(serviceName);
			}
		} catch (JAXRException je) {
			throw new RegistryException(je.getLocalizedMessage(), je);
		}
		return serviceNames;
	}
	/**
	 * Find Services based on a category ("transformation").
	 * 
	 * @param serviceType
	 * @return collection services
	 */
	public List<String> findServices(String category) throws RegistryException
	{
		List<String>serviceNames = new ArrayList<String>();
		try {
			Collection<Service>services = findServicesForCategory(category);
			for (Iterator<Service> i=services.iterator();i.hasNext();) {
				String serviceName = i.next().getName().getValue();
				serviceNames.add(serviceName);
			}
		} catch (JAXRException je) {
			throw new RegistryException(je.getLocalizedMessage(), je);
		}
		return serviceNames;
	}
	/**
	 * 
	 * @param service
	 * @return
	 */
	public List<EPR> findEPRs(String category, String serviceName) throws RegistryException, ServiceNotFoundException
	{
		List<EPR> eprs = new ArrayList<EPR>();
		Connection connection = JAXRRegistryImpl.getConnection();
		try {
			Service service = findService(category, serviceName);
			if (service==null){
				throw new ServiceNotFoundException("Could not find service with category=" + category + " and serviceName=" + serviceName);
			}
			// Get registry service and business query manager
			Collection<ServiceBinding> serviceBindings = findServiceBindings(service);
			//Converting them to EPRs
			for (Iterator i=serviceBindings.iterator();i.hasNext();) {
				ServiceBinding serviceBinding = (ServiceBinding) i.next();
				@SuppressWarnings("unused")
				String eprXML = serviceBinding.getAccessURI();
                EPR epr = null;
                //for backwards compatibility still have the decoder if
                //unmarchalling fails
                try {
                    epr = EPRHelper.fromXMLString(eprXML);
                    eprs.add(epr);
                } catch (UnmarshalException unme) {
                    try {
                        eprXML = URLDecoder.decode(eprXML, "UTF-8");
                        epr = EPRHelper.fromXMLString(eprXML);
                        eprs.add(epr);
                    } catch (UnsupportedEncodingException ue) {
                        logger.error(ue.getMessage(), ue);
                        throw new UnmarshalException(ue.getMessage(), ue);
                    }
                }
			}
		} catch (JAXRException je) {
			throw new RegistryException(je.getLocalizedMessage(), je);
        } catch (UnmarshalException me) {
            throw new RegistryException(me.getLocalizedMessage(), me);
		} finally {
			closeConnection(connection);
		}
		return eprs;
	}
	/**
	 * 
	 * @param service
	 * @return
	 */
	public EPR findEPR(String category, String serviceName) throws RegistryException, ServiceNotFoundException
	{
		EPR epr = null;
		Connection connection = JAXRRegistryImpl.getConnection();
		try {
			Service service = findService(category, serviceName);
			if (service==null){
				throw new ServiceNotFoundException("Could not find service with category=" + category + " and serviceName=" + serviceName);
			}
			// Get registry service and business query manager
			Collection<ServiceBinding> serviceBindings = findServiceBindings(service);
			if (serviceBindings.iterator().hasNext()) {
				ServiceBinding serviceBinding = (ServiceBinding) serviceBindings.iterator().next();
				String eprXML = serviceBinding.getAccessURI();
				// for backwards compatibility still have the decoder if
                // unmarchalling fails
                try {
                    epr = EPRHelper.fromXMLString(eprXML);
                } catch (UnmarshalException unme) {
                    try {
                        eprXML = URLDecoder.decode(eprXML, "UTF-8");
                        epr = EPRHelper.fromXMLString(eprXML);
                    } catch (UnsupportedEncodingException ue) {
                        logger.error(ue.getMessage(), ue);
                        throw new UnmarshalException(ue.getMessage(), ue);
                    }
                }
			}
		} catch (JAXRException je) {
			throw new RegistryException(je.getLocalizedMessage(), je);
        } catch (UnmarshalException me) {
            throw new RegistryException(me.getLocalizedMessage(), me);
		} finally {
			closeConnection(connection);
		}
		return epr;
	}
	/**
	 * Find all Organizations with a name mathing the queryString parameter.
	 * 
	 * @param organizationName used to match with the name of the organization.
	 * @return the Organization.
	 */
	protected synchronized static Organization findOrganization(String organizationName) throws JAXRException
	{
	    if (organizationName==null) {
	    	organizationName="";
	    }
		Connection connection = JAXRRegistryImpl.getConnection();
		try {
			// Get registry service and business query manager
			RegistryService rs = connection.getRegistryService();
			BusinessQueryManager bqm = rs.getBusinessQueryManager();
			// Define find qualifiers and name patterns
			Collection<String> findQualifiers = new ArrayList<String>();
			findQualifiers.add(FindQualifier.SORT_BY_NAME_ASC);
			Collection<String> namePatterns = new ArrayList<String>();
			namePatterns.add("%" + organizationName + "%");
	        //Find based upon qualifier type and values
			logger.log(Level.DEBUG,"Going to query the registry for name pattern " + namePatterns);
			BulkResponse response = bqm.findOrganizations(findQualifiers,
					namePatterns, null, null, null, null);
			if (response.getStatus()==JAXRResponse.STATUS_SUCCESS) {
				for (Iterator orgIter = response.getCollection().iterator(); orgIter.hasNext();) 
				{
					Organization org = (Organization) orgIter.next();
					logger.log(Level.DEBUG, "Organization name: " + org.getName().getValue());
					logger.log(Level.DEBUG, "Description: " + org.getDescription().getValue());
					logger.log(Level.DEBUG, "Key id: " + org.getKey().getId());
					User primaryContact = org.getPrimaryContact();
					logger.log(Level.DEBUG, "Primary Contact: " + primaryContact.getPersonName().getFullName());
					if (orgIter.hasNext()) {
						logger.log(Level.ERROR, "Found " + response.getCollection().size() 
								+ " Organization, while expecting only one of name " + organizationName);
					}
					return org;
				}
			}
			return null;
		} finally {
			closeConnection(connection);
		}
	}
	/**
	 * 
	 * @param category
	 * @param serviceName
	 * @return Service
	 */
	protected synchronized static Service findService(String category, String serviceName) throws JAXRException
	{
		if (category==null) {
			category="";
		}
		if (serviceName==null) {
			serviceName="";
		}
		Connection connection = JAXRRegistryImpl.getConnection();
		try {
			// Get registry service and business query manager
			RegistryService rs = connection.getRegistryService();
			BusinessQueryManager bqm = rs.getBusinessQueryManager();
            BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
			// Define find qualifiers and name patterns
            Collection<String> findQualifiers = new ArrayList<String>();
            findQualifiers.add(FindQualifier.AND_ALL_KEYS);
            findQualifiers.add(FindQualifier.EXACT_NAME_MATCH);
            findQualifiers.add(FindQualifier.SORT_BY_NAME_DESC);
            ClassificationScheme cScheme = bqm.findClassificationSchemeByName(findQualifiers, "org.jboss.soa.esb.:category");
            Collection<Classification> classifications = new ArrayList<Classification>();
            Classification classification = 
                blm.createClassification( 
                  cScheme, 
                  "category", category );
            classifications.add(classification);
			Collection<String> namePatterns = new ArrayList<String>();
			namePatterns.add(serviceName);
	        //Find based upon qualifier type and values
			logger.log(Level.DEBUG, "Going to query the registry for name pattern " + namePatterns);
			BulkResponse response = bqm.findServices(null, findQualifiers,
					namePatterns, classifications, null);
			if (response.getStatus()==JAXRResponse.STATUS_SUCCESS) {
				for (Iterator servIter = response.getCollection().iterator(); servIter.hasNext();) 
				{
					Service service = (Service) servIter.next();
					logger.log(Level.DEBUG, "Service name: " + service.getName().getValue());
					if (service.getDescription()!=null) {
						logger.log(Level.DEBUG, "Description: " + service.getDescription().getValue());
					}
					logger.log(Level.DEBUG, "Key id: " + service.getKey().getId());
					if (servIter.hasNext()) {
						logger.log(Level.ERROR, "Found " + response.getCollection().size() 
								+ " Services, while expecting only one by the name of " 
								+ serviceName + " in category " + category);
					}
					return service;
				}
			}
			return null;
		} finally {
			closeConnection(connection);
		}
	}
	/**
	 * Finds all services for a given category.
	 * @param category
	 * @param serviceName
	 * @return Service
	 */
	@SuppressWarnings("unchecked")
	protected synchronized static Collection<Service> findServicesForCategory(String category) throws JAXRException
	{
		Collection<Service> services = new ArrayList<Service>();
		if (category==null) {
			category="";
		}
		Connection connection = JAXRRegistryImpl.getConnection();
		try {
			// Get registry service and business query manager
			RegistryService rs = connection.getRegistryService();
			BusinessQueryManager bqm = rs.getBusinessQueryManager();
            BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
            Collection<String> findQualifiers = new ArrayList<String>();
            findQualifiers.add(FindQualifier.AND_ALL_KEYS);
            findQualifiers.add(FindQualifier.SORT_BY_NAME_DESC);
            ClassificationScheme cScheme = bqm.findClassificationSchemeByName(findQualifiers, "org.jboss.soa.esb.:category");  
            //Create classification
            Classification classification = 
                blm.createClassification( 
                  cScheme, 
                  "category", category );
            
			// Define find qualifiers and name patterns
			//Collection<String> findQualifiers = new ArrayList<String>();
			//findQualifiers.add(FindQualifier.AND_ALL_KEYS);
			findQualifiers.add(FindQualifier.SORT_BY_NAME_DESC);
			Collection<Classification> classifications = new ArrayList<Classification>();
            classifications.add(classification);
	        //Find based upon qualifier type and values
			logger.log(Level.DEBUG, "Going to query the registry for classification " + classifications);
			BulkResponse response = bqm.findServices(null, findQualifiers,
					null, classifications, null);
			if (response.getStatus()==JAXRResponse.STATUS_SUCCESS) {
				services = (Collection<Service>) response.getCollection();
			}
			return services;
		} finally {
			closeConnection(connection);
		}
	}
	/**
	 * Create a jbossesb organization under which we will register all our services.
	 * 
	 * @param blm
	 * @return
	 * @throws JAXRException
	 */
	protected synchronized static Organization createJBossESBOrganization()
			throws JAXRException 
	{
//		Getting the connection to the Registry (reading config)
		Connection connection = JAXRRegistryImpl.getConnection();
		try {
			//Logging in
			connection.setCredentials(JAXRRegistryImpl.creds);
			RegistryService rs = connection.getRegistryService();
			//Building organization
			BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
			Organization organization = blm.createOrganization(blm.createInternationalString("Red Hat/JBossESB"));
				organization.setDescription(blm.createInternationalString("Red Hat/JBoss Professional Open Source, Services for the JBossESB"));

			User user = blm.createUser();
			organization.setPrimaryContact(user);
			PersonName personName = blm.createPersonName("JBossESB");
			TelephoneNumber telephoneNumber = blm.createTelephoneNumber();
			telephoneNumber.setNumber("404 467-8555");
			telephoneNumber.setType(null);
			PostalAddress address = blm.createPostalAddress("3340",
					"Peachtree Road, NE, Suite 1200", "Atlanta", "GA", "USA",
					"30326", "");
			Collection<PostalAddress> postalAddresses = new ArrayList<PostalAddress>();
			postalAddresses.add(address);
			Collection<EmailAddress> emailAddresses = new ArrayList<EmailAddress>();
			EmailAddress emailAddress = blm
					.createEmailAddress("jbossesb@jboss.com");
			emailAddresses.add(emailAddress);
	
			Collection<TelephoneNumber> numbers = new ArrayList<TelephoneNumber>();
			numbers.add(telephoneNumber);
			user.setPersonName(personName);
			user.setPostalAddresses(postalAddresses);
			user.setEmailAddresses(emailAddresses);
			user.setTelephoneNumbers(numbers);
	
	//		Scout does not support this (yet), so leaving it out for now.
	//		ClassificationScheme cScheme = getClassificationScheme(blm,
	//				blm.createInternationalString("uddi-org:general_keywords"), blm.createInternationalString(""));
	//		Classification classification = blm.createClassification(cScheme,
	//				blm.createInternationalString("JBoss ESB"), blm.createInternationalString("JBESB"));
	//		org.addClassification(classification);
			
			saveRegistryObject(organization);
			return organization;
		} finally {
			closeConnection(connection);
		}	
	}
	/**
	 * Save Registry Object
	 */
	protected synchronized static void saveRegistryObject(RegistryObject registryObject) throws JAXRException
	{
//		Getting the connection to the Registry (reading config)
		Connection connection = JAXRRegistryImpl.getConnection();
		try {
			
			BulkResponse br = null;
			//Logging in
			connection.setCredentials(JAXRRegistryImpl.creds);
			RegistryService rs = connection.getRegistryService();
			//Building organization
			BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
			
			Class[] interfaces = registryObject.getClass().getInterfaces();
			String interfaceName ="";
			for (int i=0; i<interfaces.length; i++) {
				interfaceName = interfaces[i].getName();
				if (interfaceName.equals("javax.xml.registry.infomodel.Organization")) {
                    Collection<Organization> orginations = new ArrayList<Organization>();
                    orginations.add((Organization) registryObject);
					br = blm.saveOrganizations(orginations);
					break;
				} else if (interfaceName.equals("javax.xml.registry.infomodel.Service")) {
                    Collection<Service> services = new ArrayList<Service>();
                    services.add((Service) registryObject);
					br = blm.saveServices(services);
					break;
				} else if (interfaceName.equals("javax.xml.registry.infomodel.ServiceBinding")) {
                    Collection<ServiceBinding> serviceBindings = new ArrayList<ServiceBinding>();
                    serviceBindings.add((ServiceBinding) registryObject);
                    br = blm.saveServiceBindings(serviceBindings);
					break;
				} else {
					logger.log(Level.ERROR, "Trying to save an unsupported RegistryObject");
					throw new JAXRException("Trying to save an unsupported RegistryObject");
				}
			}
            //Verify the return
			if (br!=null && br.getStatus() == JAXRResponse.STATUS_SUCCESS) {
				logger.log(Level.DEBUG, interfaceName + " successfully saved");
				Collection coll = br.getCollection();
				Iterator iter = coll.iterator();
				while (iter.hasNext()) {
					Key key = (Key) iter.next();
					registryObject.setKey(key);
					logger.log(Level.DEBUG, "Saved Key=" + key.getId());
				}
			} else {
				logger.log(Level.ERROR, "Errors occurred during save.");
				if (br!=null) {
					Collection exceptions = br.getExceptions();
					Iterator iter = exceptions.iterator();
					String errors = "";
					JAXRException je = new JAXRException("JAXRExceptions occurred during save");
					while (iter.hasNext()) {
						Exception e = (Exception) iter.next();
						errors += e.getLocalizedMessage() + "\n";
						je.setStackTrace(e.getStackTrace());
						logger.log(Level.ERROR, e.getLocalizedMessage(), e);
						//if it's the last error, throw it now and set the current stacktrace
						if (!iter.hasNext()) {
							throw new JAXRException(errors, e);
						}
					}
					throw new JAXRException("Errors occurred during save. Response status=" + br.getStatus());
				}
				throw new JAXRException("Errors occurred during save");
			}
		} finally {
			JAXRRegistryImpl.closeConnection(connection);
		}	
	}
	/**
	 * finds the JBossESB Organizationa and creates one if it is not there.
	 * @return JBossESB Organization
	 * @throws JAXRException
	 */
	private Organization getJBossESBOrganization() throws JAXRException
	{
		jbossESBOrganization = findOrganization("Red Hat/JBossESB");
		if (jbossESBOrganization==null) {
			jbossESBOrganization = createJBossESBOrganization();
		}
		return jbossESBOrganization;
	}
	/**
	 * Find the ServiceBindings for a given Service
	 * @param service
	 * @return
	 * @throws RegistryException
	 */
	@SuppressWarnings("unchecked")
	private Collection<ServiceBinding> findServiceBindings(Service service) throws RegistryException
	{
		Collection<ServiceBinding> serviceBindings = new ArrayList<ServiceBinding>();
		Connection connection = JAXRRegistryImpl.getConnection();
		try {
			RegistryService rs = connection.getRegistryService();
			BusinessQueryManager bqm = rs.getBusinessQueryManager();
			Collection<String> findQualifiers = new ArrayList<String>();
			findQualifiers.add(FindQualifier.SORT_BY_NAME_ASC);
			BulkResponse bulkResponse = bqm.findServiceBindings(service.getKey(),findQualifiers,null,null);
			if (bulkResponse.getStatus()==JAXRResponse.STATUS_SUCCESS){
				serviceBindings = (Collection<ServiceBinding>)bulkResponse.getCollection();
			}
			return serviceBindings;
		} catch (Exception je) {
			throw new RegistryException(je.getLocalizedMessage(), je);
		} finally {
			closeConnection(connection);
		}
	}
	
}

	
