package org.jboss.internal.soa.esb.message.format.xml.marshal;

import java.net.URI;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;
import java.util.Map.Entry;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import org.jboss.internal.soa.esb.message.format.xml.XMLUtil;
import org.jboss.internal.soa.esb.util.stax.ElementContent;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.internal.soa.esb.util.stax.URIElement;
import org.jboss.soa.esb.MarshalException;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.util.ClassUtil;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * Used to plug in new Object marshal/unmarshal formats dynamically. When packing
 * objects in XML, the system runs through the list of registered plug-ins until it
 * finds one that can deal with the object type (or faults). When packing, the name (type)
 * of the plug-in that packed the object is also attached to facilitate unpacking.
 *  
 * @author Mark Little
 *
 */

public class MarshalUnmarshalManager
{
	public static MarshalUnmarshalManager getInstance ()
	{
		return _instance;
	}
	
	private MarshalUnmarshalManager ()
	{
		/*
		 * Go through the properties loaded from the property file. Anything
		 * starting with MessagePlugin.MESSAGE_PLUGIN is assumed to be a plugin
		 * that we load and add to the list.
		 */
		
		Properties properties = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperties();
		
		if (properties != null)
		{
			Enumeration names = properties.propertyNames();

			while (names.hasMoreElements())
			{
				String attrName = (String) names.nextElement();
				
				if (attrName.startsWith(MarshalUnmarshalPlugin.MARSHAL_UNMARSHAL_PLUGIN))
				{
					try
					{
						String pluginName = properties.getProperty(attrName);
						Class c = ClassUtil.forName(pluginName, getClass());
						MarshalUnmarshalPlugin thePlugin = (MarshalUnmarshalPlugin) c.newInstance();

						_plugins.put(thePlugin.type(), thePlugin);
					}
					catch (ClassNotFoundException ex)
					{
						ex.printStackTrace();
					}
					catch (IllegalAccessException ex)
					{
						ex.printStackTrace();
					}
					catch (InstantiationException ex)
					{
						ex.printStackTrace();
					}
				}
			}
			
			/*
			 * Add in the default plugin.
			 */
			
			SerializedMarshalUnmarshalPlugin defaultPlugin = new SerializedMarshalUnmarshalPlugin();
			
			_plugins.put(defaultPlugin.type(), defaultPlugin);
		}
	}
	
	/**
	 * Pack the provided object into the document.
	 * 
	 * @param out the XML stream writer.
	 * @param param the object to pack.
	 * 
	 * @return <code>true</code> if the object was packed, <code>false</code> otherwise.
	 * @throws MarshalException thrown if there is a problem packing.
	 */
	
	public boolean marshal (XMLStreamWriter out, MarshalValueImpl param) throws XMLStreamException
	{
		if ((out == null) || (param == null))
			throw new IllegalArgumentException();
		
		final String serialisedForm = param.getMarshalledForm() ;
		if (serialisedForm != null)
		{
                    final MarshalImpl marshal = new MarshalImpl(param.getMarshalledType(), serialisedForm) ;
                    StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_BODY_CONTENT_MARSHAL, marshal) ;
                    return true ;
		}
		
		final Object obj = param.getValue() ;
		for(Entry<URI, MarshalUnmarshalPlugin> entry: _plugins.entrySet())
		{
		    final MarshalUnmarshalPlugin plugin = entry.getValue() ;
		    if (plugin.canPack(obj))
		    {
                        final String value ;
                        try
                        {
                            value = plugin.marshal(obj) ;
                        }
                        catch (final MarshalException me)
                        {
                            throw new XMLStreamException(me) ;
                        }
                        param.setMarshalledForm(value) ;
                        param.setMarshalledType(plugin.type()) ;
                        
		        final MarshalImpl marshal = new MarshalImpl(entry.getKey(), value) ;
		        StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_BODY_CONTENT_MARSHAL, marshal) ;
                        
                        return true ;
		    }
		}
		
		return false;
	}
	
	/**
	 * Unpack the object from the document.
	 * 
	 * @param in the XML Stream reader.
	 * 
	 * @return the object, or <code>null</code> if this implementation cannot deal with the
	 * format.
	 * @throws XMLStreamException thrown if there is a problem unpacking.
	 */
	
	public MarshalValueImpl unmarshal (XMLStreamReader in) throws XMLStreamException
	{
	    StreamHelper.checkNextStartTag(in, XMLUtil.ESB_QNAME_BODY_CONTENT_MARSHAL) ;
	    final MarshalImpl marshal = new MarshalImpl(in) ;
	    StreamHelper.checkEndTag(in, XMLUtil.ESB_QNAME_BODY_CONTENT_MARSHAL) ;
	    
	    final MarshalUnmarshalPlugin plugin = _plugins.get(marshal.getType()) ;
	    if (plugin == null)
	    {
	        throw new XMLStreamException("Unrecognised marshal type: " + marshal.getType()) ;
	    }
	    
	    return new MarshalValueImpl(marshal.getType(), marshal.getContent()) ;
	}
	
	MarshalUnmarshalPlugin getPlugin(final URI type)
	{
            return _plugins.get(type) ;
	}
	
	private static class MarshalImpl extends ElementContent
	{
	    private URI type ;
	    private String content ;
	    
	    MarshalImpl(final URI type, final String content)
	    {
	        this.type = type ;
	        this.content = content ;
	    }
	    
	    MarshalImpl(final XMLStreamReader in)
	        throws XMLStreamException
	    {
	        super.parse(in) ;
	    }
	    
	    URI getType()
	    {
	        return type ;
	    }
	    
	    String getContent()
	    {
	        return content ;
	    }
	    
	    @Override
	    protected void putElement(XMLStreamReader in, QName elementName)
	            throws XMLStreamException
	    {
	        if (XMLUtil.ESB_QNAME_BODY_CONTENT_MARSHAL_TYPE.equals(elementName))
	        {
	            final URIElement typeURI = new URIElement(in) ;
	            type = typeURI.getValue() ;
	        }
	        else
	        {
	            throw new XMLStreamException("Unrecognised element: " + elementName) ;
	        }
	    }
	    
	    @Override
	    protected void putValue(XMLStreamReader in, String value)
	            throws XMLStreamException
	    {
	        content = value ;
	    }
	    
	    @Override
	    protected void writeChildContent(XMLStreamWriter out)
	            throws XMLStreamException
	    {
                final URIElement typeURI = new URIElement(type) ;
	        StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_BODY_CONTENT_MARSHAL_TYPE, typeURI) ;
	        out.writeCharacters(content) ;
	    }
	}
	private Hashtable<URI,MarshalUnmarshalPlugin> _plugins = new Hashtable<URI,MarshalUnmarshalPlugin>();

	private static final MarshalUnmarshalManager _instance = new MarshalUnmarshalManager();
}
