package org.jboss.soa.eclipse.wizards;

import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.jface.resource.ImageDescriptor;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;

import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;
import org.eclipse.ui.part.ISetSelectionTarget;

/**
 * This is a new wizard which creates a very simple ESB project.    The
 * wizard copies over the requisite files, sets the classpath, and refreshes.
 * Using the default WizardNewProjectCreationPage for now because all
 * that is needed to start is a project name.     If we want to customize
 * the class names or the package names we're going to have to extend that a bit.
 */
public class ESBNewWizard extends Wizard implements INewWizard {
	private WizardNewProjectCreationPage m_page;
	private IProject m_project;
	private IWorkbench m_workbench;
	
	/**
	 * Constructor for ESBNewWizard.
	 */
	public ESBNewWizard() {
		super();
		setNeedsProgressMonitor(true);
	}
	
	/**
	 * Adding the default new project wizard page to the wizard.
	 */	
	public void addPages() {
		super.addPages();
		setWindowTitle("New ESB Package Project");
		m_page = new WizardNewProjectCreationPage("WizardNewProjectCreationPage");
		m_page.setTitle("ESB Package Project");
		m_page.setDescription("Create a new ESB package project");
		m_page.setImageDescriptor(ImageDescriptor.createFromFile(getClass(),
				"/icons/sample.gif"));
		addPage(m_page);
	}

	/**
	 * Create the project, start and complete the monitor progress.
	 */
	private void createProject(IProgressMonitor monitor) throws InterruptedException, CoreException, IOException
	{
		if (monitor == null) {
			monitor= new NullProgressMonitor();
		}
		try {		
			String strName = m_page.getProjectName();
			monitor.beginTask("Creating "+ strName + " Forrest Project", 3);
	
			IProject project= m_page.getProjectHandle();
			IPath locationPath= m_page.getLocationPath();
			
			// create the project
			IProjectDescription desc= project.getWorkspace().newProjectDescription(project.getName());
			if (!m_page.useDefaults()) {
				desc.setLocation(locationPath);
			}

			ESBProjectCreator esbpc = new ESBProjectCreator();
			esbpc.createJavaProject(m_page, project, m_workbench, monitor);
		} finally {
			monitor.done();
		}
	}
	
	/**
	 * performFinish is called when the user hits the "Finish" button.
	 */
	public boolean performFinish() {
		m_project = m_page.getProjectHandle();
		WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
			protected void execute(IProgressMonitor monitor)
					throws CoreException, InterruptedException {
				try {
					createProject(monitor);
					selectAndReveal(m_project, m_workbench.getActiveWorkbenchWindow());
				} catch (IOException e) {
					e.printStackTrace();
				} finally {
					monitor.done();
				}
			}
		};
		try {
			getContainer().run(false, true, op);
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		selectAndReveal(m_project, m_workbench.getActiveWorkbenchWindow());
		return true;
	}
	
	/**
	 * Refresh the workbench.
	 * @param resource resource
	 * @param window window
	 */
    private void selectAndReveal(IResource resource,
	           IWorkbenchWindow window) {
		if (!inputValid(resource, window)) return;   
		Iterator itr = getParts(window.getActivePage()).iterator();
		while (itr.hasNext()) {
		    selectAndRevealTarget(
					window, 
					new StructuredSelection(resource), 
					getTarget((IWorkbenchPart)itr.next()));
		}
	}
	
	/**
	 * Return a target from a IWorkbenchPart.
	 * @param part workbench part
	 * @return target
	 */
    private ISetSelectionTarget getTarget(IWorkbenchPart part) {
        ISetSelectionTarget target = null;
        if (part instanceof ISetSelectionTarget) {
            target = (ISetSelectionTarget)part;
        }
        else {
            target = (ISetSelectionTarget)part.getAdapter(ISetSelectionTarget.class);
        }
		return target;		
	}
    
    /**
     * Check whether the window / resource combination is valid.
     * @param resource
     * @param window
     * @return
     */
	private boolean inputValid(IResource resource, IWorkbenchWindow window) {
		if (window == null || resource == null) return false;
		else if (window.getActivePage() == null) return false;
		else return true;
	}

	/**
	 * Call select reveal.
	 * @param window window
	 * @param selection selection
	 * @param target target
	 */private void selectAndRevealTarget(IWorkbenchWindow window, final ISelection selection, ISetSelectionTarget target) {
		if (target == null) return;
		final ISetSelectionTarget finalTarget = target;
		window.getShell().getDisplay().asyncExec(new Runnable() {
		    public void run() {
		        finalTarget.selectReveal(selection);
		    }
		});
	}
    
	 /**
	  * Return a list of parts to iterate over.
	  * @param page
	  * @return list of parts
	  */
	private List getParts(IWorkbenchPage page) {
		ArrayList result = new ArrayList();
		addParts(result, page.getViewReferences());
		addParts(result, page.getEditorReferences());
		return result;
	}
	
	/**
	 * Copy all of the workbench part references into the list.
	 * @param parts list of parts
	 * @param refs workbench part references
	 */
	private void addParts(ArrayList parts, IWorkbenchPartReference[] refs) {
		for (int i = 0; i < refs.length; i++) {
           IWorkbenchPart part = refs[i].getPart(false);
           if (part != null) {
               parts.add(part);
           }
	    }		
	}
	
	/**
	 * We will accept the selection in the workbench to see if
	 * we can initialize from it.
	 * @see IWorkbenchWizard#init(IWorkbench, IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		this.m_workbench = workbench;
		this.m_project = null;
		setNeedsProgressMonitor(true);
	}
}