package org.jbpm.gd.jpdl.ui.properties;

import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.ConfigInfoElement;
import org.jbpm.gd.jpdl.model.Delegation;
import org.jbpm.gd.jpdl.ui.dialog.ChooseDelegationClassDialog;
import org.jbpm.gd.jpdl.ui.util.ProjectFinder;

public class DelegationConfigurationComposite implements SelectionListener, FocusListener {
	
	public static DelegationConfigurationComposite create(TabbedPropertySheetWidgetFactory widgetFactory, Composite parent) {
		DelegationConfigurationComposite result = new DelegationConfigurationComposite();
		result.widgetFactory = widgetFactory;
		result.parent = parent;
		result.create();
		return result;
	}

	private ChooseDelegationClassDialog chooseDelegationClassDialog;
	
	private TabbedPropertySheetWidgetFactory widgetFactory;
	private Composite parent;

	private Label delegationClassNameLabel;
	private Text delegationClassNameText;
	private Button delegationClassNameButton;
	private Label delegationConfigTypeLabel;
	private CCombo delegationConfigTypeCombo;
	
	private Label messageLabel;
	private Label delegationConfigInfoLabel;
	
	private DelegationConfigurationText constructorDelegationConfigurationText;
	private DelegationConfigurationText compatibilityDelegationConfigurationText;
	private DelegationConfigurationTable fieldDelegationConfigurationTable;
	private DelegationConfigurationTable beanDelegationConfigurationTable;
	
	private Delegation delegation;
	
	private IType type;
	private String selectedConfiguration = "Field";
	
	private DelegationConfigurationComposite() {}
	
	private void create() {
		createDelegationClassField();
		createDelegationConfigTypeField();
		createMessageField();	
		createConfigInfoField();
		initializeLayouts();
	}
	
	private void createConfigInfoField() {
		beanDelegationConfigurationTable = DelegationConfigurationTable.create(widgetFactory, this, DelegationConfigurationTable.BEAN_POLICY);
		fieldDelegationConfigurationTable = DelegationConfigurationTable.create(widgetFactory, this, DelegationConfigurationTable.FIELD_POLICY);
		compatibilityDelegationConfigurationText = DelegationConfigurationText.create(widgetFactory, this, DelegationConfigurationText.COMPATIBILITY_POLICY);
		constructorDelegationConfigurationText = DelegationConfigurationText.create(widgetFactory, this, DelegationConfigurationText.CONSTRUCTOR_POLICY);
		delegationConfigInfoLabel = widgetFactory.createLabel(parent, "Configuration Info");
	}

	private void initializeLayouts() {
		delegationClassNameLabel.setLayoutData(createHandlerClassNameLabelLayoutData());
		delegationClassNameText.setLayoutData(createHandlerClassNameTextLayoutData());
		delegationClassNameButton.setLayoutData(createHandlerClassNameButtonLayoutData());
		delegationConfigTypeLabel.setLayoutData(createHandlerConfigTypeLabelLayoutData());
		delegationConfigTypeCombo.setLayoutData(createHandlerConfigTypeComboLayoutData());
		delegationConfigInfoLabel.setLayoutData(createHandlerConfigInfoLabelLayoutData());
		fieldDelegationConfigurationTable.setLayoutData(createConfigInfoLayoutData());
		beanDelegationConfigurationTable.setLayoutData(createConfigInfoLayoutData());
		constructorDelegationConfigurationText.setLayoutData(createConfigInfoLayoutData());
		compatibilityDelegationConfigurationText.setLayoutData(createConfigInfoLayoutData());
	}

	private void createMessageField() {
		messageLabel = widgetFactory.createLabel(parent, "");
		messageLabel.setVisible(false);
		messageLabel.setLayoutData(createMessageLabelLayoutData());
	}

	private void createDelegationConfigTypeField() {
		delegationConfigTypeLabel = widgetFactory.createLabel(parent, "Configuration Type");
		delegationConfigTypeCombo = widgetFactory.createCCombo(parent, SWT.READ_ONLY | SWT.BORDER);
		delegationConfigTypeCombo.setItems(getConfigurationTypes());
	}

	private void createDelegationClassField() {
		delegationClassNameLabel = widgetFactory.createLabel(parent, "Handler Class Name");
		delegationClassNameText = widgetFactory.createText(parent, "");
		delegationClassNameButton = widgetFactory.createButton(parent, "Search", SWT.PUSH);
		delegationClassNameButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				String className = chooseDelegationClassDialog.openDialog();
				if (className != null) {
					delegationClassNameText.setText(className);
				}
			}
		});
	}
	
	private void refreshDelegationInfo() {
		if (delegation != null) {
			type = getClassFor(delegation.getClassName());
		}
		if (type == null) {
			handleInvalidType();
		} else {
			beanDelegationConfigurationTable.refreshDelegationInfo(type);
			fieldDelegationConfigurationTable.refreshDelegationInfo(type);
			constructorDelegationConfigurationText.refreshDelegationInfo();
			compatibilityDelegationConfigurationText.refreshDelegationInfo();
		}
	}
	
	private void handleInvalidType() {
		delegationConfigInfoLabel.setVisible(false);
		fieldDelegationConfigurationTable.setVisible(false);
		beanDelegationConfigurationTable.setVisible(false);
		constructorDelegationConfigurationText.setVisible(false);
		compatibilityDelegationConfigurationText.setVisible(false);
		messageLabel.setText("The class does not exist on the project classpath.");
		messageLabel.setVisible(true);
	}
	
	private void updateConfigType() {
		String configType = delegationConfigTypeCombo.getText();
		if (configType.equals(selectedConfiguration)) return;
		selectedConfiguration = configType;
		if (delegation != null) {
			removeDelegationConfigInfo();
			delegation.setConfigInfoString(null);
			delegation.setConfigType(toConfigType(configType));
		}
		if (type == null) return;
		fieldDelegationConfigurationTable.setVisible("Field".equals(configType));
		beanDelegationConfigurationTable.setVisible("Bean".equals(configType));
		constructorDelegationConfigurationText.setVisible("Constructor".equals(configType));
		compatibilityDelegationConfigurationText.setVisible("Compatibility".equals(configType));
	}
	
	private void removeDelegationConfigInfo() {
		delegation.setConfigInfoString(null);
		ConfigInfoElement[] configInfoElements = delegation.getConfigInfoElements();
		for (int i = 0; i < configInfoElements.length; i++) {
			delegation.removeConfigInfoElement(configInfoElements[i]);
		}
	}
		
	private String toConfigType(String configType) {
		if ("Field".equals(configType)) return "field";
		if ("Bean".equals(configType)) return "bean";
		if ("Constructor".equals(configType)) return "constructor";
		if ("Compatibility".equals(configType)) return "configuration-property";
		return null;
	}
	
	private String fromConfigType(String configType) {
		if ("field".equals(configType)) return "Field";
		if ("bean".equals(configType)) return "Bean";
		if ("constructor".equals(configType)) return "Constructor";
		if ("configuration-property".equals(configType)) return "Compatibility";
		return null;
	}
	
	private FormData createConfigInfoLayoutData() {
		FormData data = new FormData();
		data.top = new FormAttachment(delegationConfigTypeCombo, 5);
		data.left = new FormAttachment(delegationConfigTypeCombo, 0);
		data.left.alignment = SWT.LEFT;
		data.right = new FormAttachment(100, -5);
		data.bottom = new FormAttachment(100, -5);
		return data;
	}
	
	private FormData createHandlerClassNameLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 5);
		data.top = new FormAttachment(0, 5);
		data.bottom = new FormAttachment(delegationClassNameText, -4);
		data.bottom.alignment = SWT.BOTTOM;
		return data;
	}
	
	private FormData createHandlerClassNameTextLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(delegationClassNameLabel, 5);
		data.right = new FormAttachment(delegationClassNameButton, -5);
		data.top = new FormAttachment(0, 5);
		return data;
	}
	
	private FormData createHandlerClassNameButtonLayoutData() {
		FormData data = new FormData();
		data.top = new FormAttachment(0, 4);
		data.right = new FormAttachment(100, -5);
		return data;
	}
	
	private FormData createHandlerConfigTypeLabelLayoutData() {
		FormData data = new FormData();
		data.top = new FormAttachment(delegationClassNameText, 5);
		data.left = new FormAttachment(0, 5);
		data.bottom = new FormAttachment(delegationConfigTypeCombo, -4);
		data.bottom.alignment = SWT.BOTTOM;
		return data;
	}
	
	private FormData createHandlerConfigTypeComboLayoutData() {
		FormData data = new FormData();
		data.top = new FormAttachment(delegationClassNameText, 5);
		data.left = new FormAttachment(delegationClassNameText, 0);
		data.left.alignment = SWT.LEFT;
		data.right = new FormAttachment(100, -5);
		return data;
	}
	
	private FormData createHandlerConfigInfoLabelLayoutData() {
		FormData data = new FormData();
		data.top = new FormAttachment(delegationConfigTypeCombo, 7);
		data.left = new FormAttachment(0, 5);
		return data;
	}
	
	private FormData createMessageLabelLayoutData() {
		FormData data = new FormData();
		data.top = new FormAttachment(delegationConfigTypeCombo, 7);
		data.left = new FormAttachment(0, 5);
		return data;
	}
	
	private String[] getConfigurationTypes() {
		return new String[] { "Field", "Bean", "Constructor", "Compatibility" };
	}
	
	public void setDelegation(Delegation newDelegation) {
		unhookListeners();
		delegation = newDelegation;
		updateControls();
		if (delegation != null) {
			hookListeners();
		}
	}

	private void hookListeners() {
		delegationClassNameText.addSelectionListener(this);
		delegationClassNameText.addFocusListener(this);
		delegationConfigTypeCombo.addSelectionListener(this);
	}

	private void unhookListeners() {
		delegationClassNameText.removeSelectionListener(this);
		delegationClassNameText.removeFocusListener(this);
		delegationConfigTypeCombo.removeSelectionListener(this);
	}
	
	private void reinitializeControls() {
		delegationClassNameText.setText("");
		delegationConfigTypeCombo.setText("Field");
		beanDelegationConfigurationTable.removeAll();
		fieldDelegationConfigurationTable.removeAll();
		constructorDelegationConfigurationText.refreshDelegationInfo();
		compatibilityDelegationConfigurationText.refreshDelegationInfo();
	}
	
	public void updateDelegationInfo(Delegation delegation) {
		if (this.delegation == null) {
			setDelegation(delegation);
		} else {
			delegation = this.delegation;
		}
		delegation.setClassName(delegationClassNameText.getText());
		String configString = delegationConfigTypeCombo.getText();
		delegation.setConfigType(toConfigType(configString));
		if ("Field".equals(configString)) {
			fieldDelegationConfigurationTable.updateDelegationConfigInfo();
		} else if ("Bean".equals(configString)) {
			beanDelegationConfigurationTable.updateDelegationConfigInfo();
		} else if ("Constructor".equals(configString)) {
			constructorDelegationConfigurationText.refreshDelegationInfo();
		} else if ("Compatibility".equals(configString)) {
			compatibilityDelegationConfigurationText.refreshDelegationInfo();
		}		
	}
	
	private void updateControls() {
		if (delegation == null) {
			reinitializeControls();
			return;
		}
		if (delegation.getClassName() != null) {
			delegationClassNameText.setText(delegation.getClassName());
		}
		if (delegation.getConfigType() != null) {
			delegationConfigTypeCombo.setText(fromConfigType(delegation.getConfigType()));
		}
		refreshDelegationInfo();
	}

	private IType getClassFor(String className) {
		if (className == null) return null;
		try {
			return ProjectFinder.getCurrentProject().findType(className);
		} catch (JavaModelException e) {
			e.printStackTrace();
			return null;
		}
	}
	
	public void setChooseDelegationClassDialog(ChooseDelegationClassDialog dialog) {
		chooseDelegationClassDialog = dialog;
	}

	private void updateClassName() {
		removeDelegationConfigInfo();
		delegation.setClassName(delegationClassNameText.getText());
		refreshDelegationInfo();
	}

	public Composite getParent() {
		return parent;
	}
	
	public Delegation getDelegation() {
		return delegation;
	}
	
	public Label getMessageLabel() {
		return messageLabel;
	}
	
	public Label getConfigInfoLabel() {
		return delegationConfigInfoLabel;
	}
	
	public void widgetDefaultSelected(SelectionEvent e) {
		if (e.widget == delegationClassNameText) {
			updateClassName();			
		}		
	}

	public void widgetSelected(SelectionEvent e) {
		if (e.widget == delegationConfigTypeCombo) {			
			updateConfigType();
		}		
	}

	public void focusGained(FocusEvent e) {
	}

	public void focusLost(FocusEvent e) {
		if (e.widget == delegationClassNameText) {
			updateClassName();			
		}		
	}

}
