/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.tools;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import org.jbpm.bpel.xml.ProblemCounter;
import org.jbpm.bpel.xml.ProblemHandler;
import org.jbpm.bpel.xml.util.XmlUtil;
import org.jbpm.jpdl.xml.Problem;

/**
 * Adapter for the <code>wscompile</code> tool, part of the Java Web Services Development Pack.
 * @author Alejandro Guizar
 * @version $Revision: 1.2 $ $Date: 2007/09/10 23:44:45 $
 */
public class WscompileTool implements JavaMappingTool {

  private String command = DEFAULT_COMMAND;
  private String packageName = DEFAULT_PACKAGE_NAME;
  private File wsdlFile = DEFAULT_WSDL_FILE;

  private File classesDirectory = DEFAULT_CLASSES_DIRECTORY;
  private File jaxrpcMappingFile = DEFAULT_JAXRPC_MAPPING_FILE;

  private ProblemHandler problemHandler = new ProblemCounter();

  static final String NS_WSCOMPILE = "http://java.sun.com/xml/ns/jax-rpc/ri/config";
  static final String ELEM_CONFIGURATION = "configuration";
  static final String ELEM_WSDL = "wsdl";
  static final String ATTR_LOCATION = "location";
  static final String ATTR_PACKAGE_NAME = "packageName";

  static final String DEFAULT_COMMAND = getDefaultCommand();
  static final String DEFAULT_PACKAGE_NAME = "org.example";
  static final File DEFAULT_WSDL_FILE = new File(WsdlServiceTool.DEFAULT_WSDL_DIRECTORY,
      WsdlServiceTool.DEFAULT_SERVICE_FILE_NAME);

  static final File DEFAULT_CLASSES_DIRECTORY = new File("classes/");
  static final File DEFAULT_JAXRPC_MAPPING_FILE = new File("jaxrpc-mapping.xml");

  private static final Log log = LogFactory.getLog(WscompileTool.class);

  /**
   * Returns the command used to invoke the <code>wscompile</code> tool.
   * @return the <code>wscompile</code> command
   */
  public String getCommand() {
    return command;
  }

  /**
   * Sets the command used to invoke the <code>wscompile</code> tool.
   * @param command the <code>wscompile</code> command
   */
  public void setCommand(String command) {
    this.command = command;
  }

  /**
   * Returns the input WSDL file.
   * @return the input WSDL file
   */
  public File getWsdlFile() {
    return wsdlFile;
  }

  public void setWsdlFile(File wsdlFile) {
    if (wsdlFile == null)
      throw new IllegalArgumentException("wsdl file cannot be null");
    this.wsdlFile = wsdlFile;
  }

  /**
   * Returns the package of generated classes.
   * @return the generated classes package
   */
  public String getPackageName() {
    return packageName;
  }

  public void setPackageName(String packageName) {
    if (packageName == null)
      throw new IllegalArgumentException("package name cannot be null");
    this.packageName = packageName;
  }

  /**
   * Returns where to place generated classes.
   * @return the generated classes directory
   */
  public File getClassesDirectory() {
    return classesDirectory;
  }

  public void setClassesDirectory(File classesDirectory) {
    if (classesDirectory == null)
      throw new IllegalArgumentException("classes directory cannot be null");
    this.classesDirectory = classesDirectory;
  }

  /**
   * Returns where to write the JAX-RPC mapping file.
   * @return the JAX-RPC mapping file
   */
  public File getJaxrpcMappingFile() {
    return jaxrpcMappingFile;
  }

  public void setJaxrpcMappingFile(File jaxrpcMappingFile) {
    if (jaxrpcMappingFile == null)
      throw new IllegalArgumentException("jax-rpc mapping file cannot be null");
    this.jaxrpcMappingFile = jaxrpcMappingFile;
  }

  public ProblemHandler getProblemHandler() {
    return problemHandler;
  }

  public void setProblemHandler(ProblemHandler problemHandler) {
    if (problemHandler == null)
      throw new IllegalArgumentException("problem handler cannot be null");
    this.problemHandler = problemHandler;
  }

  public void generateJavaMapping() {
    try {
      Document configurationDocument = generateConfiguration();
      File configurationFile = File.createTempFile("wscompile", ".xml");
      try {
        // write configuration document to file
        XmlUtil.writeFile(configurationDocument, configurationFile);
        log.info("wrote wscompile configuration: " + configurationFile);
        // invoke tool
        callWscompile(configurationFile);
      }
      finally {
        configurationFile.delete();
      }
    }
    catch (IOException e) {
      problemHandler.add(new Problem(Problem.LEVEL_ERROR, "", e));
    }
  }

  protected Document generateConfiguration() {
    Document configurationDocument = XmlUtil.createDocument();

    // configuration
    Element configuration = configurationDocument.createElementNS(NS_WSCOMPILE, ELEM_CONFIGURATION);
    XmlUtil.addNamespaceDeclaration(configuration, NS_WSCOMPILE);

    // wsdl
    Element wsdl = configurationDocument.createElementNS(NS_WSCOMPILE, ELEM_WSDL);
    wsdl.setAttribute(ATTR_LOCATION, getWsdlFile().getAbsolutePath());
    wsdl.setAttribute(ATTR_PACKAGE_NAME, packageName);

    // assemble document
    configurationDocument.appendChild(configuration);
    configuration.appendChild(wsdl);

    return configurationDocument;
  }

  protected void callWscompile(File configurationFile) throws IOException {
    // Wscompile wscompile = new Wscompile();
    String[] args = new String[10];
    args[0] = command;

    // wscompile.setImport(true);
    args[1] = "-import";

    // wscompile.setFeatures("norpcstructures,wsi");
    args[2] = "-f:norpcstructures";
    args[3] = "-f:wsi";

    // wscompile.setBase(getClassesDirectory());
    args[4] = "-d";
    args[5] = classesDirectory.getAbsolutePath();
    // ensure directory exists
    classesDirectory.mkdirs();

    // wscompile.setMapping(new File(getWebDirectory(), "jaxrpc-mapping.xml"));
    args[6] = "-mapping";
    args[7] = jaxrpcMappingFile.getAbsolutePath();

    // wscompile.setVerbose(true);
    args[8] = "-verbose";

    // wscompile.setConfig(getJavaMappingConfigurationFile());
    args[9] = configurationFile.getAbsolutePath();

    /*
     * the input and error streams are read in separate threads because wscompile may intermix
     * writes to the output stream with writes to the error stream; if only one thread read both
     * streams, it might block on the wrong stream
     */
    final Process wscompile = Runtime.getRuntime().exec(args, null,
        configurationFile.getParentFile());

    // read the error stream in separate thread
    // make sure to start the new thread before the current thread reads the input stream
    new Thread() {

      public void run() {
        try {
          logError(wscompile.getErrorStream());
        }
        catch (IOException e) {
          problemHandler.add(new Problem(Problem.LEVEL_WARNING, "wscompile error pipe broken", e));
        }
      }
    }.start();

    // read the input stream
    try {
      logInfo(wscompile.getInputStream());
    }
    catch (IOException e) {
      problemHandler.add(new Problem(Problem.LEVEL_WARNING, "wscompile input pipe broken", e));
    }

    try {
      int code = wscompile.waitFor();
      if (code == 0)
        log.info("wscompile succeeded");
      else
        problemHandler.add(new Problem(Problem.LEVEL_ERROR, "wscompile failed: code=" + code));
    }
    catch (InterruptedException e) {
      problemHandler.add(new Problem(Problem.LEVEL_ERROR, "wscompile got interrupted"));
    }
  }

  private static String getDefaultCommand() {
    return System.getProperty("os.name").toLowerCase().indexOf("windows") == -1 ? "wscompile.sh"
        : "wscompile.bat";
  }

  private static void logInfo(InputStream infoStream) throws IOException {
    try {
      BufferedReader reader = new BufferedReader(new InputStreamReader(infoStream));
      for (String line = reader.readLine(); line != null; line = reader.readLine())
        log.info(line);
    }
    finally {
      infoStream.close();
    }
  }

  private static void logError(InputStream errorStream) throws IOException {
    try {
      BufferedReader reader = new BufferedReader(new InputStreamReader(errorStream));
      for (String line = reader.readLine(); line != null; line = reader.readLine())
        log.error(line);
    }
    finally {
      errorStream.close();
    }
  }

  public void deleteGeneratedFiles() {
    // recursively delete classes directory
    if (FileUtil.deleteDirectory(classesDirectory))
      log.info("deleted: " + classesDirectory);
    // delete mapping file
    if (jaxrpcMappingFile.delete())
      log.info("deleted: " + jaxrpcMappingFile);
  }
}
