/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.tools;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLEncoder;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.jbpm.bpel.xml.ProblemCounter;
import org.jbpm.bpel.xml.ProblemHandler;
import org.jbpm.jpdl.xml.Problem;

/**
 * @author Alejandro Guizar
 * @version $Revision: 1.4 $ $Date: 2007/06/09 23:36:36 $
 */
public class ProcessDeployer {

  private String host = "localhost";
  private int port = 8080;
  private String context = "/jbpm-bpel/deploy";

  private ProblemHandler problemHandler = new ProblemCounter();

  private static final Log log = LogFactory.getLog(ProcessDeployer.class);

  public String getHost() {
    return host;
  }

  public void setHost(String host) {
    if (host == null)
      throw new IllegalArgumentException("host cannot be null");

    this.host = host;
  }

  public int getPort() {
    return port;
  }

  public void setPort(int port) {
    if (port < -1)
      throw new IllegalArgumentException("invalid port: " + port);

    this.port = port;
  }

  public String getContext() {
    return context;
  }

  public void setContext(String context) {
    if (context == null)
      throw new IllegalArgumentException("context cannot be null");

    this.context = context;
  }

  public ProblemHandler getProblemHandler() {
    return problemHandler;
  }

  public void setProblemHandler(ProblemHandler problemHandler) {
    this.problemHandler = problemHandler;
  }

  public void deployProcess(File processFile) {
    if (processFile == null)
      throw new IllegalArgumentException("process file cannot be null");

    if (!processFile.exists()) {
      problemHandler.add(new Problem(Problem.LEVEL_ERROR, "process file not found: " + processFile));
      return;
    }

    URL deployUrl;
    try {
      deployUrl = formatDeployRequest(getProcessLocation(processFile));
    }
    catch (MalformedURLException e) {
      problemHandler.add(new Problem(Problem.LEVEL_ERROR,
          "cannot format a valid url from the given host, port and context", e));
      return;
    }

    try {
      connectTo(deployUrl);
    }
    catch (IOException e) {
      problemHandler.add(new Problem(Problem.LEVEL_ERROR, "cannot connect to deployment url", e));
      return;
    }
  }

  private String getProcessLocation(File processFile) {
    URI processLocation = processFile.toURI();
    try {
      // fix a problem with File.toURI() not appending "//" after "file:"
      processLocation = new URI(processLocation.getScheme(), "", processLocation.getPath(), null);
      log.info("process location: " + processLocation);
    }
    catch (URISyntaxException e) {
      problemHandler.add(new Problem(Problem.LEVEL_WARNING, "could not add host component "
          + "to process location URI, leaving host undefined", e));
    }
    return processLocation.toString();
  }

  private URL formatDeployRequest(String processLocation) throws MalformedURLException {
    // format file component
    try {
      processLocation = URLEncoder.encode(processLocation, "UTF-8");
    }
    catch (UnsupportedEncodingException e) {
      problemHandler.add(new Problem(Problem.LEVEL_WARNING, "UTF-8 encoding not supported, "
          + "process location will be sent decoded", e));
    }
    String file = context + "?processfile=" + processLocation;

    // create deployment URL
    URL deployUrl = new URL("http", host, port, file);
    log.info("deployment url: " + deployUrl);
    return deployUrl;
  }

  private void connectTo(URL deployUrl) throws IOException {
    // create http connection
    HttpURLConnection httpConnection = (HttpURLConnection) deployUrl.openConnection();
    try {
      // contact host
      httpConnection.connect();

      // check the response
      int responseCode = httpConnection.getResponseCode();
      log.info("response code: " + responseCode);

      if (responseCode != HttpURLConnection.HTTP_OK) {
        problemHandler.add(new Problem(Problem.LEVEL_ERROR, "deployment failed, "
            + "see details in the server log file"));
      }
    }
    finally {
      httpConnection.disconnect();
    }
  }
}
